# Sidebar Navigation - Quick Start Guide

Quick reference guide for implementing and using role-based sidebar navigation in the Laravel SaaS Influencer Marketing Platform.

---

## 🚀 Quick Start

### 1. Using Sidebars in Your Views

Choose the appropriate layout based on user role:

```blade
<!-- Super Admin -->
<x-superadmin-layout>
    <x-slot name="header">Page Title</x-slot>
    <!-- Content -->
</x-superadmin-layout>

<!-- Brand Admin -->
<x-brand-admin-layout>
    <x-slot name="header">Page Title</x-slot>
    <!-- Content -->
</x-brand-admin-layout>

<!-- Creator -->
<x-creator-admin-layout>
    <x-slot name="header">Page Title</x-slot>
    <!-- Content -->
</x-creator-admin-layout>

<!-- Agency Admin -->
<x-agency-admin-layout>
    <x-slot name="header">Page Title</x-slot>
    <!-- Content -->
</x-agency-admin-layout>
```

### 2. Adding Confirmation to Actions

Add `data-confirm` attributes to destructive actions:

```html
<a href="{{ route('brand.campaigns.destroy', $campaign) }}" 
   data-confirm="Are you sure you want to delete this campaign?"
   data-confirm-title="Delete Campaign"
   data-confirm-type="danger">
    Delete
</a>
```

**Attributes:**
- `data-confirm` - Confirmation message (required)
- `data-confirm-title` - Modal title (optional, default: "Confirm Action")
- `data-confirm-type` - Modal type: "warning" or "danger" (optional, default: "warning")

### 3. Accessing Badge APIs

```javascript
// Get badge counts
fetch('/brand/api/sidebar/badges')
    .then(res => res.json())
    .then(data => console.log(data.badges));

// Search
fetch('/brand/api/sidebar/search?query=campaign')
    .then(res => res.json())
    .then(data => console.log(data.results));
```

---

## 📋 Navigation Items by Role

### Super Admin (18 items)
Dashboard, Tenants, Agencies, Users, Roles & Permissions, Campaigns, Proposals, AI Jobs, Queue & Workers, Reels, Integrations, Billing, System Logs, Alerts, Settings, Tools, Help, Account

### Brand Admin (18 items)
Dashboard, Social Accounts, Creators, Campaigns, Proposals, Content Ideas, Scheduled Posts, Reels, Ads Manager, Analytics, Competitor Insights, Messaging, Notifications, Localization, Support Bot, Settings, Help, Account

### Creator (14 items)
Dashboard, Social Accounts, Content Ideas, Reels, Scheduled Posts, Campaigns, Proposals, Messaging, Analytics, Notifications, Earnings, Support Bot, Settings, Account

### Agency Admin (12 items)
Dashboard, Managed Brands, Campaigns, Proposals, Creators, Messaging, Analytics & Reports, Billing, Notifications, Support, Settings, Account

---

## 🎨 Color Schemes

| Role | Gradient | Active Color | Badge Colors |
|------|----------|--------------|--------------|
| Super Admin | Gray-900 → Gray-900 | Red-600 | Blue, Yellow, Red, Green |
| Brand Admin | Gray-900 → Gray-800 | Blue-600 | Yellow, Green, Red |
| Creator | Purple-900 → Indigo-800 | Pink-600 | Yellow, Green, Red, Orange |
| Agency Admin | Purple-900 → Indigo-900 | Purple-600 | Purple, Yellow, Green, Red |

---

## ⌨️ Keyboard Shortcuts

| Shortcut | Action |
|----------|--------|
| `Ctrl+K` or `Cmd+K` | Open quick search |
| `Escape` | Close modals/search |
| `Tab` | Navigate items |
| `Enter` | Activate item |

---

## 🔧 Common Tasks

### Clear Badge Cache

```bash
php artisan cache:clear
```

Or programmatically:

```php
use App\Http\Controllers\SuperAdmin\SidebarController;

SidebarController::clearCache();
```

### Test Sidebar Visibility

```php
// In your test
$user = User::factory()->create();
$user->assignRole('brand_admin');

$response = $this->actingAs($user)
    ->get(route('brand.dashboard'));

$response->assertSee('Brand Admin');
```

### Add Audit Logging

```php
use Illuminate\Support\Facades\Log;

Log::channel('audit')->info('Action performed', [
    'user_id' => auth()->id(),
    'action' => 'delete_campaign',
    'resource_id' => $campaign->id,
    'timestamp' => now()
]);
```

---

## 🐛 Troubleshooting

### Sidebar Not Showing
```bash
# Check user role
php artisan tinker
>>> auth()->user()->getRoleNames()

# Verify routes
php artisan route:list | grep sidebar
```

### Badges Not Updating
```bash
# Test API endpoint
curl -H "Accept: application/json" \
     -H "X-CSRF-TOKEN: your-token" \
     http://localhost/brand/api/sidebar/badges

# Clear cache
php artisan cache:clear
```

### Search Not Working
```bash
# Check route exists
php artisan route:list | grep sidebar.search

# Check logs
tail -f storage/logs/laravel.log
```

---

## 📱 Mobile Testing

### Test on Real Devices
1. Open on mobile browser
2. Tap hamburger menu (top-left)
3. Verify sidebar slides in
4. Tap overlay to close
5. Test all navigation items
6. Verify badges display correctly

### Responsive Breakpoints
- **Desktop**: ≥1024px (full sidebar)
- **Tablet**: 768px-1023px (collapsible)
- **Mobile**: <768px (hamburger menu)

---

## 🔐 Security Checklist

- [x] Role-based access control implemented
- [x] CSRF tokens on all API calls
- [x] Confirmation modals for destructive actions
- [x] Audit logging integration ready
- [x] XSS prevention via Blade escaping
- [x] Input sanitization on search
- [x] Rate limiting on APIs (recommended)

---

## 📊 Performance Tips

1. **Optimize Badge Queries**
   ```php
   // Use indexes
   Schema::table('proposals', function (Blueprint $table) {
       $table->index(['brand_id', 'status']);
   });
   ```

2. **Cache Badge Counts**
   ```php
   Cache::remember('brand_badges_' . $brandId, 300, function () {
       // Badge queries
   });
   ```

3. **Limit Search Results**
   ```php
   ->limit(20) // Max 20 results
   ```

---

## 🎯 Best Practices

### DO ✅
- Use layouts for consistent structure
- Add confirmation to destructive actions
- Log important actions to audit trail
- Keep badge queries lightweight
- Test on multiple browsers
- Follow accessibility guidelines

### DON'T ❌
- Don't bypass role middleware
- Don't skip confirmation modals
- Don't ignore audit logging
- Don't hardcode badge counts
- Don't forget mobile testing
- Don't skip accessibility features

---

## 📚 Related Documentation

- [Complete Implementation Guide](./COMPLETE_SIDEBAR_IMPLEMENTATION.md)
- [Super Admin Sidebar](./SUPERADMIN_SIDEBAR_SETUP.md)
- [Brand Admin Sidebar](./BRAND_ADMIN_SIDEBAR.md)
- [Creator Sidebar](./CREATOR_ADMIN_SIDEBAR.md)
- [Agency Admin Sidebar](./AGENCY_ADMIN_SIDEBAR.md)

---

## 🆘 Quick Help

**Need to add a new menu item?**
→ See [Customization Guide](./COMPLETE_SIDEBAR_IMPLEMENTATION.md#customization-guide)

**Badges not showing?**
→ Check API endpoint and clear cache

**Search not working?**
→ Verify route exists and check logs

**Mobile menu broken?**
→ Check z-index and viewport meta tag

**Confirmation modal not appearing?**
→ Verify `data-confirm` attribute and Alpine.js loaded

---

**Quick Links:**
- [File Structure](./COMPLETE_SIDEBAR_IMPLEMENTATION.md#file-structure)
- [API Reference](./COMPLETE_SIDEBAR_IMPLEMENTATION.md#api-reference)
- [Troubleshooting](./COMPLETE_SIDEBAR_IMPLEMENTATION.md#troubleshooting)
- [Testing Checklist](./COMPLETE_SIDEBAR_IMPLEMENTATION.md#testing-checklist)

---

**Last Updated**: 2025-09-30  
**Version**: 1.0.0