<?php

namespace Database\Seeders;

use App\Models\Brand;
use App\Models\SocialAccount;
use App\Models\SocialAggregate;
use App\Models\SocialPost;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class SocialDemoSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get or create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create demo brand
        $brand = Brand::updateOrCreate(
            ['name' => 'Demo Brand'],
            [
                'tenant_id' => $tenant->id,
                'slug' => 'demo-brand',
                'settings' => [],
            ]
        );
        
        // Create demo brand admin user
        $brandAdmin = User::updateOrCreate(
            ['email' => 'brand@demo.local'],
            [
                'name' => 'Demo Brand Admin',
                'password' => Hash::make('password123'),
                'tenant_id' => $tenant->id,
                'brand_id' => $brand->id,
            ]
        );
        
        // Assign brand_admin role
        $brandAdmin->assignRole('brand_admin');
        
        // Create demo social accounts for brand
        $brandAccounts = [
            [
                'provider' => 'youtube',
                'provider_user_id' => 'UC_brand_youtube',
                'meta' => [
                    'username' => 'Brand YouTube Channel',
                    'name' => 'Brand YouTube Channel',
                    'followers' => 25000,
                ],
            ],
            [
                'provider' => 'instagram',
                'provider_user_id' => 'brand_instagram',
                'meta' => [
                    'username' => 'brand_instagram',
                    'name' => 'Brand Instagram Account',
                    'followers' => 18500,
                ],
            ],
            [
                'provider' => 'twitter',
                'provider_user_id' => 'brand_twitter',
                'meta' => [
                    'username' => 'brand_twitter',
                    'name' => 'Brand Twitter Account',
                    'followers' => 5200,
                ],
            ],
            [
                'provider' => 'linkedin',
                'provider_user_id' => 'brand_linkedin',
                'meta' => [
                    'username' => 'Brand LinkedIn',
                    'name' => 'Brand LinkedIn Account',
                    'followers' => 8400,
                ],
            ],
            [
                'provider' => 'tiktok',
                'provider_user_id' => 'brand_tiktok',
                'meta' => [
                    'username' => 'brand_tiktok',
                    'name' => 'Brand TikTok Account',
                    'followers' => 32000,
                ],
            ],
        ];
        
        foreach ($brandAccounts as $accountData) {
            $account = SocialAccount::updateOrCreate(
                [
                    'brand_id' => $brand->id,
                    'provider' => $accountData['provider'],
                ],
                [
                    'tenant_id' => $brand->tenant_id,
                    'provider_user_id' => $accountData['provider_user_id'],
                    'access_token_encrypted' => 'demo_token_' . $accountData['provider'],
                    'refresh_token_encrypted' => 'demo_refresh_' . $accountData['provider'],
                    'token_expires_at' => now()->addYear(),
                    'meta' => $accountData['meta'],
                    'connected_at' => now(),
                    'last_synced_at' => now(),
                    'is_demo' => true,
                ]
            );
            
            // Create demo posts for each account
            $this->createDemoPosts($account);
        }
        
        // Create demo aggregates for brand
        $this->createDemoAggregates(null, $brand->id, $brand->tenant_id);
        
        // Create multiple demo creators with varied influencer types
        $creatorTypes = [
            'Lifestyle', 'Tech', 'Gaming', 'Beauty', 'Fitness',
            'Travel', 'Food', 'Education', 'Business',
            'Entertainment', 'Sports', 'Other'
        ];
        
        // Create 30 demo creators
        for ($i = 1; $i <= 30; $i++) {
            $type = $creatorTypes[array_rand($creatorTypes)];
            $customType = null;
            
            // For "Other" type, set a custom type
            if ($type === 'Other') {
                $customTypes = ['Photography', 'Art', 'Music', 'Dance', 'Cooking', 'Fashion', 'Comedy', 'Vlogging'];
                $customType = $customTypes[array_rand($customTypes)];
            }
            
            $user = User::updateOrCreate(
                ['email' => "creator{$i}@demo.local"],
                [
                    'name' => "Demo Creator {$i}",
                    'password' => Hash::make('password123'),
                    'tenant_id' => $tenant->id,
                    'influencer_type' => $type,
                    'influencer_type_custom' => $customType,
                    'meta' => [
                        'bio' => "This is the bio for Demo Creator {$i}. I specialize in " . ($customType ?? $type) . " content.",
                    ],
                ]
            );
            
            // Assign creator role if not already assigned
            if (!$user->hasRole('creator')) {
                $user->assignRole('creator');
            }
            
            // Create demo social accounts for creator
            $creatorAccounts = [
                [
                    'provider' => 'youtube',
                    'provider_user_id' => "UC_demo_youtube_{$i}",
                    'meta' => [
                        'username' => "Demo YouTube Channel {$i}",
                        'name' => "Demo YouTube Channel {$i}",
                        'followers' => rand(1000, 50000),
                    ],
                ],
                [
                    'provider' => 'instagram',
                    'provider_user_id' => "demo_instagram_{$i}",
                    'meta' => [
                        'username' => "demo_instagram_{$i}",
                        'name' => "Demo Instagram Account {$i}",
                        'followers' => rand(500, 30000),
                    ],
                ],
                [
                    'provider' => 'twitter',
                    'provider_user_id' => "demo_twitter_{$i}",
                    'meta' => [
                        'username' => "demo_twitter_{$i}",
                        'name' => "Demo Twitter Account {$i}",
                        'followers' => rand(100, 15000),
                    ],
                ],
                [
                    'provider' => 'linkedin',
                    'provider_user_id' => "demo_linkedin_{$i}",
                    'meta' => [
                        'username' => "Demo LinkedIn {$i}",
                        'name' => "Demo LinkedIn Account {$i}",
                        'followers' => rand(500, 20000),
                    ],
                ],
                [
                    'provider' => 'tiktok',
                    'provider_user_id' => "demo_tiktok_{$i}",
                    'meta' => [
                        'username' => "demo_tiktok_{$i}",
                        'name' => "Demo TikTok Account {$i}",
                        'followers' => rand(1000, 40000),
                    ],
                ],
            ];
            
            foreach ($creatorAccounts as $accountData) {
                $account = SocialAccount::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'provider' => $accountData['provider'],
                    ],
                    [
                        'tenant_id' => $user->tenant_id,
                        'provider_user_id' => $accountData['provider_user_id'],
                        'access_token_encrypted' => 'demo_token_' . $accountData['provider'] . '_' . $i,
                        'refresh_token_encrypted' => 'demo_refresh_' . $accountData['provider'] . '_' . $i,
                        'token_expires_at' => now()->addYear(),
                        'meta' => $accountData['meta'],
                        'connected_at' => now(),
                        'last_synced_at' => now(),
                        'is_demo' => true,
                    ]
                );
                
                // Create demo posts for each account
                $this->createDemoPosts($account);
            }
            
            // Create demo aggregates for creator
            $this->createDemoAggregates($user->id, null, $user->tenant_id);
        }
        
        // Create a sample proposal for testing
        $firstCreator = User::whereHas('roles', function ($q) {
            $q->where('name', 'creator');
        })->first();
        
        if ($firstCreator && $brandAdmin) {
            \App\Models\Proposal::updateOrCreate(
                [
                    'brand_id' => $brandAdmin->brand_id,
                    'creator_id' => $firstCreator->id,
                ],
                [
                    'title' => 'Collaboration Proposal for Summer Campaign',
                    'message' => 'We would like to propose a collaboration for our upcoming summer campaign. We believe your audience would be a perfect fit for our brand.',
                    'type' => 'Collaboration',
                    'budget' => 1500.00,
                    'duration' => '4 weeks',
                    'deliverables' => ['2 Instagram posts', '1 Instagram reel', '1 YouTube video'],
                    'status' => 'sent',
                ]
            );
        }
    }
    
    /**
     * Create demo posts for an account.
     *
     * @param SocialAccount $account
     * @return void
     */
    protected function createDemoPosts(SocialAccount $account)
    {
        $postTitles = [
            'How to Master Laravel in 30 Days',
            'Understanding Vue.js Components',
            'Building a REST API with Node.js',
            'CSS Grid vs Flexbox: When to Use Which',
            'Database Optimization Techniques',
            'Getting Started with Docker',
            'Advanced JavaScript Patterns',
            'Responsive Design Best Practices',
            'PHP 8.2 New Features Explained',
            'Creating Beautiful Admin Dashboards',
            'Social Media Marketing Strategies',
            'Content Creation Tips for Creators',
            'Building a Multi-Tenant Application',
            'Laravel Queues and Job Processing',
            'API Authentication with Sanctum',
            'Vue 3 Composition API Guide',
            'React Hooks Deep Dive',
            'Tailwind CSS Utility First Approach',
            'Testing Laravel Applications',
            'Deploying Apps with CI/CD Pipelines',
        ];
        
        $postContent = [
            'In this comprehensive guide, we\'ll explore how to master Laravel in just 30 days with practical examples and exercises.',
            'Vue.js components are the building blocks of any Vue application. Let\'s dive deep into how they work and best practices.',
            'Node.js is a powerful platform for building REST APIs. Here\'s how to get started with creating your first API endpoint.',
            'CSS Grid and Flexbox are both powerful layout tools. Understanding when to use each one is crucial for modern web design.',
            'Database optimization can significantly improve your application\'s performance. These techniques will help you get started.',
            'Docker has revolutionized how we deploy and manage applications. Let\'s explore the basics of containerization.',
            'JavaScript patterns can help you write more maintainable and scalable code. Here are some advanced patterns to know.',
            'Responsive design ensures your website looks great on all devices. These best practices will help you achieve that.',
            'PHP 8.2 introduced several new features that can improve your development workflow. Let\'s explore them in detail.',
            'Admin dashboards are crucial for business applications. Here\'s how to create beautiful and functional dashboards.',
            'Social media marketing is essential for creators. These strategies will help you grow your audience effectively.',
            'Content creation can be challenging. Here are some tips to help you produce better content consistently.',
            'Multi-tenant applications serve multiple clients from a single codebase. Here\'s how to build one with Laravel.',
            'Laravel queues allow you to defer time-consuming tasks. Let\'s explore how to use them effectively.',
            'API authentication is crucial for secure applications. Laravel Sanctum provides a simple solution.',
            'Vue 3 Composition API offers a new way to organize your component logic. Here\'s a complete guide.',
            'React Hooks allow you to use state and other React features without writing a class. Let\'s dive deep.',
            'Tailwind CSS utility-first approach can speed up your development process. Here\'s how to use it effectively.',
            'Testing is essential for maintaining code quality. Here\'s how to test Laravel applications properly.',
            'CI/CD pipelines automate your deployment process. Let\'s explore how to set one up for your application.',
        ];
        
        // Create 20 demo posts for each account
        for ($i = 0; $i < 20; $i++) {
            $publishedAt = now()->subDays(rand(0, 13))->subHours(rand(0, 23));
            
            SocialPost::updateOrCreate(
                [
                    'social_account_id' => $account->id,
                    'provider_post_id' => $account->provider . '_post_' . $i,
                ],
                [
                    'title' => $postTitles[$i] ?? 'Demo Post ' . $i,
                    'caption' => 'Demo caption for post ' . $i,
                    'content' => $postContent[$i] ?? 'This is demo content for post ' . $i,
                    'media' => [
                        'thumbnail' => 'https://picsum.photos/400/300?random=' . $i,
                    ],
                    'permalink' => 'https://example.com/' . $account->provider . '/post/' . $i,
                    'published_at' => $publishedAt,
                    'metrics' => [
                        'views' => rand(100, 5000),
                        'likes' => rand(10, 500),
                        'comments' => rand(0, 100),
                        'shares' => rand(0, 50),
                    ],
                    'raw_json' => [
                        'demo' => true,
                        'post_id' => $account->provider . '_post_' . $i,
                    ],
                    'last_checked_at' => now(),
                ]
            );
        }
    }
    
    /**
     * Create demo aggregates for the user or brand.
     *
     * @param int|null $userId
     * @param int|null $brandId
     * @return void
     */
    protected function createDemoAggregates($userId, $brandId, $tenantId)
    {
        // Create total followers aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'brand_id' => $brandId,
                'tenant_id' => $tenantId,
                'key' => 'total_followers',
            ],
            [
                'value' => [
                    'value' => $userId ? 41100 : 93100, // Sum of all demo followers
                    'previous' => $userId ? 38500 : 89500,
                ],
                'computed_at' => now(),
            ]
        );
        
        // Create total posts aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'brand_id' => $brandId,
                'tenant_id' => $tenantId,
                'key' => 'total_posts',
            ],
            [
                'value' => [
                    'value' => 100, // 5 providers * 20 posts each
                    'previous' => $userId ? 85 : 95,
                ],
                'computed_at' => now(),
            ]
        );
        
        // Create engagement rate aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'brand_id' => $brandId,
                'tenant_id' => $tenantId,
                'key' => 'engagement_rate',
            ],
            [
                'value' => [
                    'value' => $userId ? 4.25 : 5.75,
                    'previous' => $userId ? 3.80 : 5.20,
                ],
                'computed_at' => now(),
            ]
        );
        
        // Create total views aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'brand_id' => $brandId,
                'tenant_id' => $tenantId,
                'key' => 'total_views',
            ],
            [
                'value' => [
                    'value' => $userId ? 125000 : 250000,
                    'previous' => $userId ? 98000 : 210000,
                ],
                'computed_at' => now(),
            ]
        );
        
        // Create daily views aggregates for the last 14 days
        for ($i = 0; $i < 14; $i++) {
            $date = now()->subDays($i)->format('Y-m-d');
            
            SocialAggregate::updateOrCreate(
                [
                    'user_id' => $userId,
                    'brand_id' => $brandId,
                    'tenant_id' => $tenantId,
                    'key' => 'daily_views',
                    'computed_at' => $date,
                ],
                [
                    'value' => rand(5000, 25000),
                    'computed_at' => $date,
                ]
            );
        }
    }
}
