<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('shared_social_access', function (Blueprint $table) {
            $table->id();
            $table->foreignId('connection_id')->constrained('creator_social_connections')->onDelete('cascade');
            $table->foreignId('creator_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('brand_id')->constrained('brands')->onDelete('cascade');
            $table->foreignId('campaign_id')->nullable()->constrained('campaigns')->onDelete('cascade');
            
            // Access control
            $table->json('allowed_scopes'); // e.g., ['read_profile', 'read_posts', 'publish_content']
            $table->enum('access_level', ['read_only', 'read_write', 'full'])->default('read_only');
            
            // Temporary access token (encrypted, scoped)
            $table->text('scoped_access_token')->nullable();
            $table->timestamp('token_expires_at')->nullable();
            
            // Time-based access control
            $table->timestamp('granted_at');
            $table->timestamp('expires_at');
            $table->timestamp('revoked_at')->nullable();
            $table->foreignId('revoked_by')->nullable()->constrained('users')->onDelete('set null');
            $table->text('revocation_reason')->nullable();
            
            // Status tracking
            $table->enum('status', ['active', 'expired', 'revoked'])->default('active');
            $table->timestamp('last_used_at')->nullable();
            $table->integer('usage_count')->default(0);
            
            // Consent tracking
            $table->boolean('creator_consent')->default(false);
            $table->timestamp('consent_given_at')->nullable();
            $table->ipAddress('consent_ip')->nullable();
            $table->text('consent_user_agent')->nullable();
            
            // Notification tracking
            $table->boolean('creator_notified')->default(false);
            $table->boolean('brand_notified')->default(false);
            $table->boolean('expiry_warning_sent')->default(false);
            
            // Audit fields
            $table->foreignId('granted_by')->nullable()->constrained('users')->onDelete('set null');
            $table->json('audit_trail')->nullable(); // Track all access events
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['connection_id', 'brand_id']);
            $table->index(['creator_id', 'brand_id']);
            $table->index(['brand_id', 'status']);
            $table->index(['campaign_id', 'status']);
            $table->index('status');
            $table->index('expires_at');
            $table->index(['status', 'expires_at']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('shared_social_access');
    }
};