<?php

namespace App\Services\SocialProviders;

use App\Models\SocialAccount;
use Illuminate\Support\Facades\Config;

class YouTubeProvider extends BaseProvider
{
    /**
     * Configure the provider.
     *
     * @return void
     */
    protected function configure()
    {
        $this->slug = 'youtube';
        $this->name = 'YouTube';
        
        $this->endpoints = [
            'authorize' => 'https://accounts.google.com/o/oauth2/auth',
            'token' => 'https://oauth2.googleapis.com/token',
            'api' => 'https://www.googleapis.com/youtube/v3',
        ];
        
        $this->scopes = [
            'https://www.googleapis.com/auth/youtube.readonly',
            'https://www.googleapis.com/auth/userinfo.profile',
            'https://www.googleapis.com/auth/userinfo.email',
        ];
        
        $this->clientId = Config::get('services.google.client_id');
        $this->clientSecret = Config::get('services.google.client_secret');
        // Don't set redirectUri here, let the base provider handle it
    }

    /**
     * Get user info using the access token.
     *
     * @param string $token
     * @return array
     */
    protected function getUserInfo($token)
    {
        $url = 'https://www.googleapis.com/oauth2/v2/userinfo?access_token=' . urlencode($token);
        
        $response = $this->httpClient->get($url);
        return json_decode($response->getBody(), true);
    }

    /**
     * Fetch account details from the provider.
     *
     * @param SocialAccount $account
     * @return array
     */
    public function fetchAccountDetails(SocialAccount $account): array
    {
        // For demo accounts, return fake data
        if ($account->is_demo) {
            return [
                'id' => $account->provider_user_id,
                'name' => 'Demo YouTube Channel',
                'subscribers' => 15000,
                'views' => 500000,
                'videos' => 120,
            ];
        }

        try {
            // Get channel details
            $url = $this->endpoints['api'] . '/channels?part=snippet,statistics&id=' . $account->provider_user_id;
            $response = $this->makeApiRequest($account, $url);
            
            if (empty($response['items'])) {
                throw new \Exception('Channel not found');
            }
            
            $channel = $response['items'][0];
            
            return [
                'id' => $channel['id'],
                'name' => $channel['snippet']['title'],
                'subscribers' => (int) $channel['statistics']['subscriberCount'],
                'views' => (int) $channel['statistics']['viewCount'],
                'videos' => (int) $channel['statistics']['videoCount'],
                'thumbnail' => $channel['snippet']['thumbnails']['default']['url'] ?? null,
                'description' => $channel['snippet']['description'] ?? null,
            ];
        } catch (\Exception $e) {
            // Log the error and return empty array
            $this->logRequestException($e, 'Failed to fetch YouTube account details', $account);
            return [];
        }
    }

    /**
     * Fetch recent posts from the provider.
     *
     * @param SocialAccount $account
     * @param int $limit
     * @return array
     */
    public function fetchRecentPosts(SocialAccount $account, int $limit = 20): array
    {
        // For demo accounts, return fake data
        if ($account->is_demo) {
            $posts = [];
            for ($i = 1; $i <= min($limit, 10); $i++) {
                $posts[] = [
                    'id' => 'demo_video_' . $i,
                    'title' => 'Demo Video #' . $i,
                    'description' => 'This is a demo video description for video #' . $i,
                    'published_at' => now()->subDays($i)->toIso8601String(),
                    'thumbnail' => 'https://i.ytimg.com/vi/demo_video_' . $i . '/mqdefault.jpg',
                    'permalink' => 'https://www.youtube.com/watch?v=demo_video_' . $i,
                    'views' => rand(1000, 50000),
                    'likes' => rand(100, 5000),
                    'comments' => rand(10, 500),
                    'shares' => rand(50, 1000),
                ];
            }
            return $posts;
        }

        try {
            // Get videos for the channel
            $url = $this->endpoints['api'] . '/search?part=snippet&channelId=' . $account->provider_user_id . '&maxResults=' . $limit . '&order=date&type=video';
            $response = $this->makeApiRequest($account, $url);
            
            $videos = [];
            foreach ($response['items'] as $item) {
                $videoId = $item['id']['videoId'];
                
                // Get detailed video statistics
                $statsUrl = $this->endpoints['api'] . '/videos?part=statistics&id=' . $videoId;
                $statsResponse = $this->makeApiRequest($account, $statsUrl);
                
                $stats = $statsResponse['items'][0]['statistics'] ?? [];
                
                $videos[] = [
                    'id' => $videoId,
                    'title' => $item['snippet']['title'],
                    'description' => $item['snippet']['description'],
                    'published_at' => $item['snippet']['publishedAt'],
                    'thumbnail' => $item['snippet']['thumbnails']['medium']['url'] ?? null,
                    'permalink' => 'https://www.youtube.com/watch?v=' . $videoId,
                    'views' => (int) ($stats['viewCount'] ?? 0),
                    'likes' => (int) ($stats['likeCount'] ?? 0),
                    'comments' => (int) ($stats['commentCount'] ?? 0),
                    'shares' => 0, // YouTube API doesn't provide share count
                ];
            }
            
            return $videos;
        } catch (\Exception $e) {
            // Log the error and return empty array
            $this->logRequestException($e, 'Failed to fetch YouTube videos', $account);
            return [];
        }
    }
}