<?php

namespace App\Services\SocialProviders;

use InvalidArgumentException;

class ProviderFactory
{
    /**
     * Create a new provider instance.
     *
     * @param string $slug
     * @return ProviderInterface
     * @throws InvalidArgumentException
     */
    public static function make(string $slug): ProviderInterface
    {
        // Map provider slugs to their classes
        $providers = [
            'youtube' => YouTubeProvider::class,
            'instagram' => InstagramProvider::class,
            'twitter' => TwitterProvider::class,
            'linkedin' => LinkedInProvider::class,
            'tiktok' => TikTokProvider::class,
            'facebook' => InstagramProvider::class, // Facebook uses Instagram provider
            'pinterest' => PinterestProvider::class,
            'reddit' => RedditProvider::class,
        ];

        // Check if the provider is supported
        if (!isset($providers[$slug])) {
            throw new InvalidArgumentException("Unsupported provider: {$slug}");
        }

        // Get the provider class
        $providerClass = $providers[$slug];

        // Check if the class exists
        if (!class_exists($providerClass)) {
            throw new InvalidArgumentException("Provider class does not exist: {$providerClass}");
        }

        // Create and return the provider instance
        return new $providerClass();
    }

    /**
     * Get all supported provider slugs.
     *
     * @return array
     */
    public static function getSupportedProviders(): array
    {
        return [
            'youtube' => 'YouTube',
            'instagram' => 'Instagram',
            'twitter' => 'Twitter',
            'linkedin' => 'LinkedIn',
            'tiktok' => 'TikTok',
            'facebook' => 'Facebook',
            'pinterest' => 'Pinterest',
            'reddit' => 'Reddit',
        ];
    }

    /**
     * Get the provider name by slug.
     *
     * @param string $slug
     * @return string|null
     */
    public static function getProviderName(string $slug): ?string
    {
        $providers = self::getSupportedProviders();
        return $providers[$slug] ?? null;
    }
}