<?php

namespace App\Services\SocialProviders;

use App\Models\SocialAccount;
use App\Models\SocialSyncError;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Exception;

abstract class BaseProvider implements ProviderInterface
{
    /**
     * The HTTP client instance.
     *
     * @var Client
     */
    protected $httpClient;

    /**
     * The provider slug.
     *
     * @var string
     */
    protected $slug;

    /**
     * The provider name.
     *
     * @var string
     */
    protected $name;

    /**
     * The provider's OAuth endpoints.
     *
     * @var array
     */
    protected $endpoints = [];

    /**
     * The provider's OAuth scopes.
     *
     * @var array
     */
    protected $scopes = [];

    /**
     * The provider's OAuth client ID.
     *
     * @var string
     */
    protected $clientId;

    /**
     * The provider's OAuth client secret.
     *
     * @var string
     */
    protected $clientSecret;

    /**
     * The redirect URI for OAuth callbacks.
     *
     * @var string
     */
    protected $redirectUri;

    /**
     * Create a new provider instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->httpClient = new Client();
        $this->configure();
    }

    /**
     * Configure the provider.
     *
     * @return void
     */
    abstract protected function configure();

    /**
     * Redirect the user to the provider's authentication page.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function redirectToProvider(Request $request): RedirectResponse
    {
        // Generate a random state parameter for security
        $state = Str::random(40);
        $request->session()->put('social_auth_state', $state);

        // Build the authorization URL
        $url = $this->buildAuthorizationUrl($state);

        return redirect($url);
    }

    /**
     * Build the authorization URL.
     *
     * @param string $state
     * @return string
     */
    protected function buildAuthorizationUrl($state)
    {
        $params = [
            'client_id' => $this->clientId,
            'redirect_uri' => $this->getRedirectUri(),
            'scope' => implode(',', $this->scopes),
            'response_type' => 'code',
            'state' => $state,
        ];

        return $this->endpoints['authorize'] . '?' . http_build_query($params);
    }

    /**
     * Get the redirect URI, handling cases where routes might not be available.
     *
     * @return string
     */
    protected function getRedirectUri()
    {
        if ($this->redirectUri) {
            return $this->redirectUri;
        }

        // Fallback to a default URI if the route is not available
        try {
            return route('creator.connections.callback', ['provider' => $this->slug]);
        } catch (\Exception $e) {
            // Return a placeholder URI for testing purposes
            return 'http://localhost/creator/connections/callback/' . $this->slug;
        }
    }

    /**
     * Handle the callback from the provider.
     *
     * @param Request $request
     * @return array
     */
    public function handleCallback(Request $request): array
    {
        // Verify the state parameter
        $state = $request->session()->pull('social_auth_state');
        if (!$state || $state !== $request->input('state')) {
            throw new \Exception('Invalid state parameter');
        }

        // Get the authorization code
        $code = $request->input('code');
        if (!$code) {
            throw new \Exception('Authorization code not provided');
        }

        // Exchange the authorization code for an access token
        $tokenResponse = $this->exchangeCodeForToken($code);

        // Get user info using the access token
        $userResponse = $this->getUserInfo($tokenResponse['access_token']);

        return [
            'token' => $tokenResponse,
            'user' => $userResponse,
        ];
    }

    /**
     * Exchange the authorization code for an access token.
     *
     * @param string $code
     * @return array
     */
    protected function exchangeCodeForToken($code)
    {
        $params = [
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
            'redirect_uri' => $this->getRedirectUri(),
            'code' => $code,
            'grant_type' => 'authorization_code',
        ];

        try {
            $response = $this->httpClient->post($this->endpoints['token'], [
                'form_params' => $params,
            ]);

            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            $this->logRequestException($e, 'Token exchange failed');
            throw $e;
        }
    }

    /**
     * Get user info using the access token.
     *
     * @param string $token
     * @return array
     */
    abstract protected function getUserInfo($token);

    /**
     * Fetch account details from the provider.
     *
     * @param SocialAccount $account
     * @return array
     */
    abstract public function fetchAccountDetails(SocialAccount $account): array;

    /**
     * Fetch recent posts from the provider.
     *
     * @param SocialAccount $account
     * @param int $limit
     * @return array
     */
    abstract public function fetchRecentPosts(SocialAccount $account, int $limit = 20): array;

    /**
     * Refresh the access token for the account.
     *
     * @param SocialAccount $account
     * @return bool
     */
    public function refreshToken(SocialAccount $account): bool
    {
        // If there's no refresh token, we can't refresh
        $refreshToken = $account->getRefreshToken();
        if (!$refreshToken) {
            return false;
        }

        $params = [
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
            'refresh_token' => $refreshToken,
            'grant_type' => 'refresh_token',
        ];

        try {
            $response = $this->httpClient->post($this->endpoints['token'], [
                'form_params' => $params,
            ]);

            $tokenData = json_decode($response->getBody(), true);

            // Update the account with the new tokens
            $account->setAccessToken($tokenData['access_token']);
            
            if (isset($tokenData['refresh_token'])) {
                $account->setRefreshToken($tokenData['refresh_token']);
            }
            
            if (isset($tokenData['expires_in'])) {
                $account->token_expires_at = now()->addSeconds($tokenData['expires_in']);
            }
            
            $account->save();

            return true;
        } catch (RequestException $e) {
            $this->logRequestException($e, 'Token refresh failed', $account);
            return false;
        }
    }

    /**
     * Make an authenticated API request.
     *
     * @param SocialAccount $account
     * @param string $url
     * @param string $method
     * @param array $options
     * @return array
     */
    protected function makeApiRequest(SocialAccount $account, $url, $method = 'GET', $options = [])
    {
        // Check if the token is expired and try to refresh it
        if ($account->isTokenExpired()) {
            if (!$this->refreshToken($account)) {
                throw new \Exception('Token expired and refresh failed');
            }
        }

        // Get the access token
        $token = $account->getAccessToken();
        if (!$token) {
            throw new \Exception('No access token available');
        }

        // Add the authorization header
        $options['headers']['Authorization'] = 'Bearer ' . $token;

        try {
            $response = $this->httpClient->request($method, $url, $options);
            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            $this->logRequestException($e, 'API request failed', $account);
            throw $e;
        }
    }

    /**
     * Log a request exception.
     *
     * @param Exception $e
     * @param string $message
     * @param SocialAccount|null $account
     * @return void
     */
    protected function logRequestException(Exception $e, $message, $account = null)
    {
        // Log the error
        Log::error($message, [
            'exception' => $e->getMessage(),
            'response' => method_exists($e, 'hasResponse') && $e->hasResponse() ? $e->getResponse()->getBody() : null,
            'account_id' => $account ? $account->id : null,
            'provider' => $this->slug,
        ]);

        // Record the error in the database
        $error = new SocialSyncError();
        $error->social_account_id = $account ? $account->id : null;
        $error->error_text = $message . ': ' . $e->getMessage();
        $error->response_code = method_exists($e, 'hasResponse') && $e->hasResponse() ? $e->getResponse()->getStatusCode() : null;
        $error->meta = [
            'exception' => get_class($e),
            'provider' => $this->slug,
            'url' => method_exists($e, 'hasResponse') && $e->hasResponse() ? (string) $e->getRequest()->getUri() : null,
        ];
        $error->save();
    }

    /**
     * Get the provider slug.
     *
     * @return string
     */
    public function getSlug()
    {
        return $this->slug;
    }

    /**
     * Get the provider name.
     *
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }
    
    /**
     * Post content to the provider.
     *
     * @param ScheduledPost $scheduledPost
     * @param SocialAccount $account
     * @return array
     */
    public function postContent($scheduledPost, SocialAccount $account): array
    {
        // This method should be implemented by specific providers
        throw new \Exception('postContent method not implemented for ' . $this->name);
    }
}