<?php

namespace App\Services\Providers;

use App\Models\ScheduledReel;
use App\Models\SocialAccount;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class YouTubeProvider implements ReelProviderInterface
{
    /**
     * Post a reel to YouTube.
     *
     * @param ScheduledReel $reel
     * @param SocialAccount $account
     * @return array
     */
    public function postReel(ScheduledReel $reel, SocialAccount $account): array
    {
        try {
            // Check if account has valid token
            $token = $account->getDecryptedToken();
            
            if (!$token) {
                return [
                    'status' => 'failed',
                    'message' => 'Missing access token'
                ];
            }
            
            // Check if token is expired
            if ($account->isTokenExpired()) {
                // Try to refresh token if refresh token exists
                if ($refreshToken = $account->getRefreshToken()) {
                    $refreshed = $this->refreshToken($account, $refreshToken);
                    if ($refreshed) {
                        $token = $account->getDecryptedToken();
                    } else {
                        return [
                            'status' => 'failed',
                            'message' => 'Token expired and refresh failed'
                        ];
                    }
                } else {
                    return [
                        'status' => 'failed',
                        'message' => 'Token expired and no refresh token available'
                    ];
                }
            }
            
            // Get signed source URLs
            $sourceUrls = $reel->getSignedSourceUrls();
            
            if (empty($sourceUrls)) {
                return [
                    'status' => 'failed',
                    'message' => 'No source media found'
                ];
            }
            
            // For demo accounts, just return success
            if ($account->isDemo()) {
                return [
                    'status' => 'success',
                    'message' => 'Reel posted successfully (demo)',
                    'video_id' => 'demo_' . time(),
                    'permalink' => 'https://youtube.com/watch?v=demo_' . time()
                ];
            }
            
            // In a real implementation, you would:
            // 1. Download the source media
            // 2. Process it according to the template and options
            // 3. Upload it to YouTube using the YouTube Data API
            
            // For now, we'll simulate a successful post
            Log::info('Posting reel to YouTube', [
                'scheduled_reel_id' => $reel->id,
                'social_account_id' => $account->id,
                'source_count' => count($sourceUrls)
            ]);
            
            return [
                'status' => 'success',
                'message' => 'Reel posted successfully',
                'video_id' => 'video_' . time(),
                'permalink' => 'https://youtube.com/watch?v=video_' . time()
            ];
        } catch (\Exception $e) {
            Log::error('Failed to post reel to YouTube: ' . $e->getMessage(), [
                'scheduled_reel_id' => $reel->id,
                'social_account_id' => $account->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Refresh YouTube access token.
     *
     * @param SocialAccount $account
     * @param string $refreshToken
     * @return bool
     */
    protected function refreshToken(SocialAccount $account, string $refreshToken): bool
    {
        try {
            $clientId = config('services.youtube.client_id');
            $clientSecret = config('services.youtube.client_secret');
            
            $response = Http::asForm()->post('https://oauth2.googleapis.com/token', [
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
                'refresh_token' => $refreshToken,
                'grant_type' => 'refresh_token'
            ]);
            
            if ($response->successful()) {
                $data = $response->json();
                
                $account->update([
                    'access_token_encrypted' => \Crypt::encryptString($data['access_token']),
                    'token_expires_at' => now()->addSeconds($data['expires_in'] ?? 3600) // Default to 1 hour
                ]);
                
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Failed to refresh YouTube token: ' . $e->getMessage(), [
                'social_account_id' => $account->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            return false;
        }
    }
}