<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\User;
use App\Models\InfluencerMatch;
use App\Models\AiTask;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class InfluencerMatchmakingService
{
    /**
     * Get influencer recommendations for a campaign
     *
     * @param Campaign $campaign
     * @param array $filters
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getRecommendations(Campaign $campaign, array $filters = [])
    {
        // Check if we already have matches for this campaign
        $existingMatches = InfluencerMatch::where('campaign_id', $campaign->id)->count();
        
        if ($existingMatches === 0) {
            // Generate matches if none exist
            $this->generateMatches($campaign);
        }
        
        // Build query for matches
        $query = InfluencerMatch::with(['creator' => function($query) {
            $query->with('socialAccounts');
        }])
        ->where('campaign_id', $campaign->id)
        ->join('users', 'matches.creator_id', '=', 'users.id')
        ->select('matches.*');
        
        // Apply filters
        if (!empty($filters['min_followers'])) {
            $query->whereHas('creator.socialAccounts', function($q) use ($filters) {
                $q->where('followers_count', '>=', $filters['min_followers']);
            });
        }
        
        if (!empty($filters['max_followers'])) {
            $query->whereHas('creator.socialAccounts', function($q) use ($filters) {
                $q->where('followers_count', '<=', $filters['max_followers']);
            });
        }
        
        if (!empty($filters['min_engagement'])) {
            $query->whereHas('creator.socialAccounts', function($q) use ($filters) {
                $q->where('engagement_rate', '>=', $filters['min_engagement']);
            });
        }
        
        if (!empty($filters['location'])) {
            $query->where('users.location', 'LIKE', '%' . $filters['location'] . '%');
        }
        
        if (!empty($filters['niche'])) {
            $query->where('users.niche', 'LIKE', '%' . $filters['niche'] . '%');
        }
        
        // Only show creators who opted-in or are public
        $query->where(function($q) {
            $q->where('users.is_public', true)
              ->orWhere('users.opted_in', true);
        });
        
        return $query->orderBy('score', 'desc')->get();
    }
    
    /**
     * Generate matches for a campaign using AI
     *
     * @param Campaign $campaign
     * @return void
     */
    public function generateMatches(Campaign $campaign)
    {
        // Create AI task for matchmaking
        $aiTask = AiTask::create([
            'type' => 'influencer_matchmaking',
            'status' => 'pending',
            'brand_id' => $campaign->brand_id,
            'result_meta' => [
                'campaign_id' => $campaign->id,
                'campaign_objectives' => $campaign->objectives,
                'campaign_audience' => $campaign->audience,
                'campaign_budget' => $campaign->budget,
                'campaign_deliverables' => $campaign->deliverables,
            ],
            'provider' => 'openai',
        ]);
        
        // In a real implementation, this would call an AI service
        // For now, we'll simulate the AI response
        $this->processAiMatchmakingResponse($aiTask);
    }
    
    /**
     * Process the AI matchmaking response
     *
     * @param AiTask $aiTask
     * @return void
     */
    protected function processAiMatchmakingResponse(AiTask $aiTask)
    {
        $payload = $aiTask->result_meta;
        $campaignId = $payload['campaign_id'] ?? null;
        
        if (!$campaignId) {
            $aiTask->update([
                'status' => 'failed',
                'error_message' => 'Invalid campaign ID in payload',
            ]);
            return;
        }
        
        $campaign = Campaign::find($campaignId);
        if (!$campaign) {
            $aiTask->update([
                'status' => 'failed',
                'error_message' => 'Campaign not found',
            ]);
            return;
        }
        
        // Get potential creators (in a real implementation, this would be more sophisticated)
        $creators = User::where('influencer_type', '!=', null)
            ->where(function($query) {
                $query->where('is_public', true)
                      ->orWhere('opted_in', true);
            })
            ->limit(50) // Limit for performance
            ->get();
        
        // Create matches for each creator (simulated AI scoring)
        foreach ($creators as $creator) {
            // Simulate AI scoring based on campaign and creator attributes
            $score = $this->calculateMatchScore($campaign, $creator);
            $reasons = $this->generateMatchReasons($campaign, $creator, $score);
            
            // Store the match
            InfluencerMatch::updateOrCreate(
                [
                    'creator_id' => $creator->id,
                    'campaign_id' => $campaign->id,
                ],
                [
                    'score' => $score,
                    'reasons' => $reasons,
                ]
            );
        }
        
        // Update the AI task status
        $aiTask->update([
            'status' => 'completed',
            'result_text' => 'Influencer matchmaking completed successfully',
        ]);
    }
    
    /**
     * Calculate match score between campaign and creator
     *
     * @param Campaign $campaign
     * @param User $creator
     * @return float
     */
    protected function calculateMatchScore(Campaign $campaign, User $creator)
    {
        $score = 0;
        
        // Get creator's social accounts
        $socialAccounts = $creator->socialAccounts;
        
        if ($socialAccounts->isEmpty()) {
            return 0; // No social accounts, can't match
        }
        
        // Calculate average followers count
        $avgFollowers = $socialAccounts->avg('followers_count');
        
        // Calculate average engagement rate
        $avgEngagement = $socialAccounts->avg('engagement_rate');
        
        // Score based on audience match
        if ($campaign->audience) {
            $audienceMatch = $this->calculateAudienceMatch($campaign->audience, $creator);
            $score += $audienceMatch * 40; // Weight audience match at 40%
        }
        
        // Score based on niche match
        if ($campaign->deliverables) {
            $nicheMatch = $this->calculateNicheMatch($campaign->deliverables, $creator);
            $score += $nicheMatch * 30; // Weight niche match at 30%
        }
        
        // Score based on engagement rate (10-30% range is good)
        if ($avgEngagement >= 10 && $avgEngagement <= 30) {
            $score += 15; // Good engagement rate
        } elseif ($avgEngagement > 30) {
            $score += 10; // High engagement rate (may be too high)
        } else {
            $score += $avgEngagement; // Low engagement rate
        }
        
        // Score based on follower count (scaled)
        if ($avgFollowers > 0) {
            // Logarithmic scale for followers (more followers = higher score, but with diminishing returns)
            $followerScore = log10($avgFollowers) * 5;
            $score += min($followerScore, 15); // Cap at 15 points
        }
        
        return min($score, 100); // Cap at 100
    }
    
    /**
     * Calculate audience match score
     *
     * @param string $campaignAudience
     * @param User $creator
     * @return float
     */
    protected function calculateAudienceMatch($campaignAudience, User $creator)
    {
        // In a real implementation, this would use AI to analyze audience demographics
        // For now, we'll use a simple keyword matching approach
        
        if (!$campaignAudience || !$creator->audience) {
            return 0;
        }
        
        $campaignKeywords = explode(',', strtolower($campaignAudience));
        $creatorKeywords = explode(',', strtolower($creator->audience ?? ''));
        
        $matches = 0;
        foreach ($campaignKeywords as $keyword) {
            $keyword = trim($keyword);
            if (in_array($keyword, $creatorKeywords)) {
                $matches++;
            }
        }
        
        return ($matches / count($campaignKeywords)) * 100;
    }
    
    /**
     * Calculate niche match score
     *
     * @param string $campaignDeliverables
     * @param User $creator
     * @return float
     */
    protected function calculateNicheMatch($campaignDeliverables, User $creator)
    {
        // In a real implementation, this would use AI to analyze content themes
        // For now, we'll use a simple keyword matching approach
        
        if (!$campaignDeliverables || !$creator->niche) {
            return 0;
        }
        
        $campaignKeywords = explode(',', strtolower($campaignDeliverables));
        $creatorKeywords = explode(',', strtolower($creator->niche ?? ''));
        
        $matches = 0;
        foreach ($campaignKeywords as $keyword) {
            $keyword = trim($keyword);
            if (in_array($keyword, $creatorKeywords)) {
                $matches++;
            }
        }
        
        return ($matches / count($campaignKeywords)) * 100;
    }
    
    /**
     * Generate reasons for the match
     *
     * @param Campaign $campaign
     * @param User $creator
     * @param float $score
     * @return array
     */
    protected function generateMatchReasons(Campaign $campaign, User $creator, $score)
    {
        $reasons = [];
        
        // Get creator's social accounts
        $socialAccounts = $creator->socialAccounts;
        
        if ($socialAccounts->isEmpty()) {
            $reasons[] = "No social accounts found";
            return $reasons;
        }
        
        // Calculate average followers count
        $avgFollowers = $socialAccounts->avg('followers_count');
        
        // Calculate average engagement rate
        $avgEngagement = $socialAccounts->avg('engagement_rate');
        
        // Add reasons based on scoring
        if ($campaign->audience) {
            $audienceMatch = $this->calculateAudienceMatch($campaign->audience, $creator);
            if ($audienceMatch > 70) {
                $reasons[] = "Strong audience demographic match ({$audienceMatch}%)";
            } elseif ($audienceMatch > 40) {
                $reasons[] = "Good audience demographic match ({$audienceMatch}%)";
            }
        }
        
        if ($campaign->deliverables) {
            $nicheMatch = $this->calculateNicheMatch($campaign->deliverables, $creator);
            if ($nicheMatch > 70) {
                $reasons[] = "Strong content niche match ({$nicheMatch}%)";
            } elseif ($nicheMatch > 40) {
                $reasons[] = "Good content niche match ({$nicheMatch}%)";
            }
        }
        
        if ($avgEngagement >= 10 && $avgEngagement <= 30) {
            $reasons[] = "Optimal engagement rate ({$avgEngagement}%)";
        } elseif ($avgEngagement > 30) {
            $reasons[] = "High engagement rate ({$avgEngagement}%) - may have lower quality interactions";
        } else {
            $reasons[] = "Low engagement rate ({$avgEngagement}%)";
        }
        
        if ($avgFollowers > 10000) {
            $reasons[] = "Large follower base ({$avgFollowers} followers)";
        } elseif ($avgFollowers > 1000) {
            $reasons[] = "Good follower base ({$avgFollowers} followers)";
        }
        
        // If score is high, add a general reason
        if ($score > 80) {
            $reasons[] = "Overall excellent match for this campaign";
        } elseif ($score > 60) {
            $reasons[] = "Good match for this campaign";
        }
        
        return $reasons;
    }
    
    /**
     * Save a match for later reference
     *
     * @param int $creatorId
     * @param int $campaignId
     * @return InfluencerMatch
     */
    public function saveMatch($creatorId, $campaignId)
    {
        return InfluencerMatch::firstOrCreate([
            'creator_id' => $creatorId,
            'campaign_id' => $campaignId,
        ]);
    }
    
    /**
     * Invite a creator to a campaign
     *
     * @param int $creatorId
     * @param int $campaignId
     * @return void
     */
    public function inviteCreator($creatorId, $campaignId)
    {
        // In a real implementation, this would send an invitation to the creator
        // For now, we'll just log the action
        
        Log::info("Invited creator {$creatorId} to campaign {$campaignId}");
        
        // Update the match to indicate invitation sent
        InfluencerMatch::where('creator_id', $creatorId)
            ->where('campaign_id', $campaignId)
            ->update(['invited_at' => now()]);
    }
    
    /**
     * Export matches to CSV
     *
     * @param Campaign $campaign
     * @return string
     */
    public function exportMatches(Campaign $campaign)
    {
        $matches = $this->getRecommendations($campaign);
        
        $csv = "Creator Name,Email,Score,Followers,Engagement Rate,Reasons\n";
        
        foreach ($matches as $match) {
            $creator = $match->creator;
            $socialAccounts = $creator->socialAccounts;
            
            $followers = $socialAccounts->sum('followers_count');
            $engagement = $socialAccounts->avg('engagement_rate');
            $reasons = implode('; ', $match->reasons ?? []);
            
            $csv .= "\"{$creator->name}\",\"{$creator->email}\",\"{$match->score}\",\"{$followers}\",\"{$engagement}\",\"{$reasons}\"\n";
        }
        
        return $csv;
    }
}