<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\CampaignFinancial;
use App\Models\FinancialReport;
use App\Models\BudgetScenario;
use App\Models\FinancialAuditLog;
use App\Models\User;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class FinancialInsightsService
{
    private AIFinancialAnalysisService $aiService;

    public function __construct(AIFinancialAnalysisService $aiService)
    {
        $this->aiService = $aiService;
    }

    /**
     * Create or update campaign financial data
     */
    public function updateCampaignFinancials(
        Campaign $campaign,
        array $spendData,
        array $performanceData,
        Carbon $periodStart,
        Carbon $periodEnd
    ): CampaignFinancial {
        $financial = CampaignFinancial::updateOrCreate(
            [
                'campaign_id' => $campaign->id,
                'period_start' => $periodStart,
                'period_end' => $periodEnd,
            ],
            [
                'brand_id' => $campaign->brand_id,
                'ad_spend' => $spendData['ad_spend'] ?? 0,
                'influencer_payments' => $spendData['influencer_payments'] ?? 0,
                'production_costs' => $spendData['production_costs'] ?? 0,
                'other_costs' => $spendData['other_costs'] ?? 0,
                'clicks' => $performanceData['clicks'] ?? 0,
                'impressions' => $performanceData['impressions'] ?? 0,
                'conversions' => $performanceData['conversions'] ?? 0,
                'avg_order_value' => $performanceData['avg_order_value'] ?? 0,
                'total_revenue' => ($performanceData['conversions'] ?? 0) * ($performanceData['avg_order_value'] ?? 0),
                'spend_breakdown' => $spendData['breakdown'] ?? null,
                'performance_metrics' => $performanceData['metrics'] ?? null,
            ]
        );

        // Log the update
        FinancialAuditLog::logAccess(
            $financial,
            auth()->user(),
            'updated',
            'Campaign financial data updated'
        );

        return $financial;
    }

    /**
     * Generate campaign financial report
     */
    public function generateCampaignReport(
        Campaign $campaign,
        Carbon $periodStart,
        Carbon $periodEnd,
        bool $runAIAnalysis = true
    ): FinancialReport {
        // Aggregate financial data for the period
        $financials = CampaignFinancial::where('campaign_id', $campaign->id)
            ->inPeriod($periodStart, $periodEnd)
            ->get();

        $aggregated = $this->aggregateFinancialData($financials);

        // Create report
        $report = FinancialReport::create([
            'campaign_id' => $campaign->id,
            'brand_id' => $campaign->brand_id,
            'report_type' => 'campaign',
            'total_spend' => $aggregated['total_spend'],
            'total_revenue' => $aggregated['total_revenue'],
            'net_profit' => $aggregated['net_profit'],
            'roi_percentage' => $aggregated['roi_percentage'],
            'breakeven_days' => $aggregated['breakeven_days'],
            'total_clicks' => $aggregated['total_clicks'],
            'total_impressions' => $aggregated['total_impressions'],
            'total_conversions' => $aggregated['total_conversions'],
            'avg_conversion_rate' => $aggregated['avg_conversion_rate'],
            'avg_cost_per_conversion' => $aggregated['avg_cost_per_conversion'],
            'spend_breakdown' => $aggregated['spend_breakdown'],
            'revenue_breakdown' => $aggregated['revenue_breakdown'],
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'ai_status' => $runAIAnalysis ? 'pending' : 'completed',
        ]);

        // Run AI analysis if requested
        if ($runAIAnalysis) {
            $this->aiService->analyzeFinancialReport($report);
        }

        // Log report generation
        FinancialAuditLog::logAccess(
            $report,
            auth()->user(),
            'created',
            'Campaign financial report generated'
        );

        return $report;
    }

    /**
     * Generate brand-level consolidated report
     */
    public function generateBrandReport(
        User $brand,
        Carbon $periodStart,
        Carbon $periodEnd,
        bool $runAIAnalysis = true
    ): FinancialReport {
        // Get all campaign financials for the brand
        $financials = CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($periodStart, $periodEnd)
            ->get();

        $aggregated = $this->aggregateFinancialData($financials);

        // Create consolidated report
        $report = FinancialReport::create([
            'brand_id' => $brand->id,
            'report_type' => 'brand_consolidated',
            'total_spend' => $aggregated['total_spend'],
            'total_revenue' => $aggregated['total_revenue'],
            'net_profit' => $aggregated['net_profit'],
            'roi_percentage' => $aggregated['roi_percentage'],
            'breakeven_days' => $aggregated['breakeven_days'],
            'total_clicks' => $aggregated['total_clicks'],
            'total_impressions' => $aggregated['total_impressions'],
            'total_conversions' => $aggregated['total_conversions'],
            'avg_conversion_rate' => $aggregated['avg_conversion_rate'],
            'avg_cost_per_conversion' => $aggregated['avg_cost_per_conversion'],
            'spend_breakdown' => $aggregated['spend_breakdown'],
            'revenue_breakdown' => $aggregated['revenue_breakdown'],
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'ai_status' => $runAIAnalysis ? 'pending' : 'completed',
        ]);

        // Run AI analysis if requested
        if ($runAIAnalysis) {
            $this->aiService->analyzeFinancialReport($report);
        }

        // Log report generation
        FinancialAuditLog::logAccess(
            $report,
            auth()->user(),
            'created',
            'Brand consolidated financial report generated'
        );

        return $report;
    }

    /**
     * Aggregate financial data from multiple records
     */
    private function aggregateFinancialData(Collection $financials): array
    {
        $totalSpend = $financials->sum('total_spend');
        $totalRevenue = $financials->sum('total_revenue');
        $netProfit = $totalRevenue - $totalSpend;
        
        $roiPercentage = $totalSpend > 0 
            ? (($totalRevenue - $totalSpend) / $totalSpend) * 100 
            : null;

        $totalClicks = $financials->sum('clicks');
        $totalConversions = $financials->sum('conversions');
        
        $avgConversionRate = $totalClicks > 0 
            ? ($totalConversions / $totalClicks) * 100 
            : 0;

        $avgCostPerConversion = $totalConversions > 0 
            ? $totalSpend / $totalConversions 
            : 0;

        // Calculate average breakeven days
        $breakevenDays = $financials->whereNotNull('breakeven_days')
            ->avg('breakeven_days');

        // Aggregate spend breakdown by channel
        $spendBreakdown = [];
        $revenueBreakdown = [];
        
        foreach ($financials as $financial) {
            if ($financial->spend_breakdown) {
                foreach ($financial->spend_breakdown as $channel => $amount) {
                    $spendBreakdown[$channel] = ($spendBreakdown[$channel] ?? 0) + $amount;
                }
            }
            
            if ($financial->performance_metrics && isset($financial->performance_metrics['revenue_by_channel'])) {
                foreach ($financial->performance_metrics['revenue_by_channel'] as $channel => $amount) {
                    $revenueBreakdown[$channel] = ($revenueBreakdown[$channel] ?? 0) + $amount;
                }
            }
        }

        return [
            'total_spend' => $totalSpend,
            'total_revenue' => $totalRevenue,
            'net_profit' => $netProfit,
            'roi_percentage' => $roiPercentage,
            'breakeven_days' => $breakevenDays ? (int) round($breakevenDays) : null,
            'total_clicks' => $totalClicks,
            'total_impressions' => $financials->sum('impressions'),
            'total_conversions' => $totalConversions,
            'avg_conversion_rate' => $avgConversionRate,
            'avg_cost_per_conversion' => $avgCostPerConversion,
            'spend_breakdown' => $spendBreakdown,
            'revenue_breakdown' => $revenueBreakdown,
        ];
    }

    /**
     * Get financial dashboard data for brand
     */
    public function getDashboardData(User $brand, ?Carbon $periodStart = null, ?Carbon $periodEnd = null): array
    {
        $periodStart = $periodStart ?? now()->subDays(30);
        $periodEnd = $periodEnd ?? now();

        // Get latest consolidated report or generate one
        $report = FinancialReport::where('brand_id', $brand->id)
            ->where('report_type', 'brand_consolidated')
            ->where('period_start', '>=', $periodStart)
            ->where('period_end', '<=', $periodEnd)
            ->latest()
            ->first();

        if (!$report) {
            $report = $this->generateBrandReport($brand, $periodStart, $periodEnd);
        }

        // Get campaign-level reports
        $campaignReports = FinancialReport::where('brand_id', $brand->id)
            ->where('report_type', 'campaign')
            ->where('period_start', '>=', $periodStart)
            ->where('period_end', '<=', $periodEnd)
            ->with('campaign')
            ->get();

        // Get budget scenarios
        $scenarios = BudgetScenario::where('brand_id', $brand->id)
            ->where('status', '!=', 'archived')
            ->latest()
            ->take(3)
            ->get();

        // Calculate trends
        $trends = $this->calculateTrends($brand, $periodStart, $periodEnd);

        // Get top performing campaigns
        $topCampaigns = $this->getTopPerformingCampaigns($brand, $periodStart, $periodEnd, 5);

        return [
            'overview' => [
                'total_spend' => $report->total_spend,
                'total_revenue' => $report->total_revenue,
                'net_profit' => $report->net_profit,
                'roi_percentage' => $report->roi_percentage,
                'roi_status' => $report->getRoiStatus(),
                'total_conversions' => $report->total_conversions,
                'avg_conversion_rate' => $report->avg_conversion_rate,
            ],
            'report' => $report,
            'campaign_reports' => $campaignReports,
            'scenarios' => $scenarios,
            'trends' => $trends,
            'top_campaigns' => $topCampaigns,
            'insights' => $report->actionable_insights ?? [],
            'recommendations' => $report->recommendations_text,
        ];
    }

    /**
     * Calculate financial trends
     */
    private function calculateTrends(User $brand, Carbon $periodStart, Carbon $periodEnd): array
    {
        $currentPeriodDays = $periodStart->diffInDays($periodEnd);
        $previousPeriodStart = $periodStart->copy()->subDays($currentPeriodDays);
        $previousPeriodEnd = $periodStart->copy()->subDay();

        // Get current period data
        $currentData = CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($periodStart, $periodEnd)
            ->get();

        // Get previous period data
        $previousData = CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($previousPeriodStart, $previousPeriodEnd)
            ->get();

        $currentMetrics = $this->aggregateFinancialData($currentData);
        $previousMetrics = $this->aggregateFinancialData($previousData);

        return [
            'spend_trend' => $this->calculatePercentageChange(
                $previousMetrics['total_spend'],
                $currentMetrics['total_spend']
            ),
            'revenue_trend' => $this->calculatePercentageChange(
                $previousMetrics['total_revenue'],
                $currentMetrics['total_revenue']
            ),
            'roi_trend' => $this->calculatePercentageChange(
                $previousMetrics['roi_percentage'] ?? 0,
                $currentMetrics['roi_percentage'] ?? 0
            ),
            'conversion_trend' => $this->calculatePercentageChange(
                $previousMetrics['total_conversions'],
                $currentMetrics['total_conversions']
            ),
        ];
    }

    /**
     * Calculate percentage change
     */
    private function calculatePercentageChange(float $old, float $new): array
    {
        if ($old == 0) {
            return [
                'value' => $new > 0 ? 100 : 0,
                'direction' => $new > 0 ? 'up' : 'neutral',
            ];
        }

        $change = (($new - $old) / $old) * 100;

        return [
            'value' => round($change, 2),
            'direction' => $change > 0 ? 'up' : ($change < 0 ? 'down' : 'neutral'),
        ];
    }

    /**
     * Get top performing campaigns
     */
    private function getTopPerformingCampaigns(
        User $brand,
        Carbon $periodStart,
        Carbon $periodEnd,
        int $limit = 5
    ): Collection {
        return CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($periodStart, $periodEnd)
            ->with('campaign')
            ->orderByDesc('roi_percentage')
            ->take($limit)
            ->get();
    }

    /**
     * Generate budget scenarios for a report
     */
    public function generateScenarios(FinancialReport $report, float $totalBudget): array
    {
        return $this->aiService->generateBudgetScenarios($report, $totalBudget);
    }

    /**
     * Apply a budget scenario
     */
    public function applyScenario(BudgetScenario $scenario): bool
    {
        DB::beginTransaction();
        
        try {
            // Mark scenario as applied
            $scenario->markAsApplied();

            // Deactivate other scenarios for the same brand
            BudgetScenario::where('brand_id', $scenario->brand_id)
                ->where('id', '!=', $scenario->id)
                ->where('is_applied', true)
                ->update(['is_applied' => false]);

            // Log the action
            FinancialAuditLog::logAccess(
                $scenario,
                auth()->user(),
                'scenario_applied',
                "Applied {$scenario->scenario_type} budget scenario"
            );

            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            return false;
        }
    }

    /**
     * Export financial report
     */
    public function exportReport(FinancialReport $report, string $format = 'pdf'): array
    {
        // Increment export count
        $report->incrementExportCount();

        // Generate filename
        $filename = sprintf(
            'financial_report_%s_%s.%s',
            $report->id,
            now()->format('Y-m-d'),
            $format
        );

        // Log export
        FinancialAuditLog::logAccess(
            $report,
            auth()->user(),
            'exported',
            "Report exported as {$format}",
            [
                'export_format' => $format,
                'export_filename' => $filename,
            ]
        );

        return [
            'filename' => $filename,
            'format' => $format,
            'data' => $report->toArray(),
        ];
    }

    /**
     * Get financial insights summary
     */
    public function getInsightsSummary(User $brand): array
    {
        $latestReport = FinancialReport::where('brand_id', $brand->id)
            ->where('report_type', 'brand_consolidated')
            ->latest()
            ->first();

        if (!$latestReport || !$latestReport->isAiProcessed()) {
            return [
                'has_insights' => false,
                'message' => 'No insights available yet. Generate a financial report to get AI-powered insights.',
            ];
        }

        return [
            'has_insights' => true,
            'insights_count' => $latestReport->insights_count,
            'insights' => $latestReport->actionable_insights,
            'recommendations' => $latestReport->recommendations_text,
            'roi_status' => $latestReport->getRoiStatus(),
            'generated_at' => $latestReport->ai_processed_at,
        ];
    }

    /**
     * Compare two financial periods
     */
    public function comparePeriods(
        User $brand,
        Carbon $period1Start,
        Carbon $period1End,
        Carbon $period2Start,
        Carbon $period2End
    ): array {
        $period1Data = CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($period1Start, $period1End)
            ->get();

        $period2Data = CampaignFinancial::where('brand_id', $brand->id)
            ->inPeriod($period2Start, $period2End)
            ->get();

        $period1Metrics = $this->aggregateFinancialData($period1Data);
        $period2Metrics = $this->aggregateFinancialData($period2Data);

        return [
            'period1' => [
                'start' => $period1Start,
                'end' => $period1End,
                'metrics' => $period1Metrics,
            ],
            'period2' => [
                'start' => $period2Start,
                'end' => $period2End,
                'metrics' => $period2Metrics,
            ],
            'comparison' => [
                'spend_change' => $this->calculatePercentageChange(
                    $period1Metrics['total_spend'],
                    $period2Metrics['total_spend']
                ),
                'revenue_change' => $this->calculatePercentageChange(
                    $period1Metrics['total_revenue'],
                    $period2Metrics['total_revenue']
                ),
                'roi_change' => $this->calculatePercentageChange(
                    $period1Metrics['roi_percentage'] ?? 0,
                    $period2Metrics['roi_percentage'] ?? 0
                ),
            ],
        ];
    }
}