<?php

namespace App\Services;

use App\Models\CompetitorBrand;
use App\Models\CompetitorAlert;
use App\Models\CompetitorPost;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class CompetitorAlertService
{
    /**
     * Check for performance spikes and create alerts.
     *
     * @param CompetitorBrand $competitor
     * @return array
     */
    public function checkForAlerts(CompetitorBrand $competitor): array
    {
        $alerts = [];

        try {
            // Check for engagement rate spikes
            $engagementAlerts = $this->checkEngagementSpikes($competitor);
            $alerts = array_merge($alerts, $engagementAlerts);

            // Check for follower growth (if data available)
            $followerAlerts = $this->checkFollowerGrowth($competitor);
            $alerts = array_merge($alerts, $followerAlerts);

            // Check for posting frequency changes
            $frequencyAlerts = $this->checkPostingFrequency($competitor);
            $alerts = array_merge($alerts, $frequencyAlerts);

            // Check for viral posts
            $viralAlerts = $this->checkViralPosts($competitor);
            $alerts = array_merge($alerts, $viralAlerts);

        } catch (\Exception $e) {
            Log::error("Failed to check alerts for competitor {$competitor->id}: {$e->getMessage()}");
        }

        return $alerts;
    }

    /**
     * Check for engagement rate spikes.
     *
     * @param CompetitorBrand $competitor
     * @return array
     */
    protected function checkEngagementSpikes(CompetitorBrand $competitor): array
    {
        $alerts = [];
        
        // Get recent posts (last 7 days)
        $recentPosts = $competitor->posts()
            ->where('published_at', '>=', now()->subDays(7))
            ->get();

        if ($recentPosts->isEmpty()) {
            return $alerts;
        }

        // Get historical average (30-60 days ago)
        $historicalPosts = $competitor->posts()
            ->whereBetween('published_at', [now()->subDays(60), now()->subDays(30)])
            ->get();

        if ($historicalPosts->isEmpty()) {
            return $alerts;
        }

        $recentAvg = $recentPosts->avg('engagement_rate');
        $historicalAvg = $historicalPosts->avg('engagement_rate');

        // Calculate percentage change
        $changePercentage = $historicalAvg > 0 
            ? (($recentAvg - $historicalAvg) / $historicalAvg) * 100 
            : 0;

        // Create alert if spike is significant (>20% increase)
        if ($changePercentage > 20) {
            $severity = $this->determineSeverity($changePercentage, [20, 50, 100]);
            
            $alert = CompetitorAlert::create([
                'brand_id' => $competitor->brand_id,
                'competitor_brand_id' => $competitor->id,
                'tenant_id' => $competitor->tenant_id,
                'alert_type' => 'spike',
                'metric' => 'engagement_rate',
                'severity' => $severity,
                'title' => "Engagement Rate Spike Detected",
                'description' => "Competitor {$competitor->name} has seen a {$changePercentage}% increase in engagement rate over the past week.",
                'data' => [
                    'recent_avg' => round($recentAvg, 2),
                    'historical_avg' => round($historicalAvg, 2),
                    'change_percentage' => round($changePercentage, 2),
                ],
                'threshold_value' => round($historicalAvg, 2),
                'current_value' => round($recentAvg, 2),
                'change_percentage' => round($changePercentage, 2),
            ]);

            $alerts[] = $alert;
        }

        return $alerts;
    }

    /**
     * Check for follower growth changes.
     *
     * @param CompetitorBrand $competitor
     * @return array
     */
    protected function checkFollowerGrowth(CompetitorBrand $competitor): array
    {
        $alerts = [];
        
        // This would require tracking follower counts over time
        // Placeholder for future implementation when follower data is available
        
        return $alerts;
    }

    /**
     * Check for posting frequency changes.
     *
     * @param CompetitorBrand $competitor
     * @return array
     */
    protected function checkPostingFrequency(CompetitorBrand $competitor): array
    {
        $alerts = [];

        // Get recent posting frequency (last 7 days)
        $recentPosts = $competitor->posts()
            ->where('published_at', '>=', now()->subDays(7))
            ->count();

        // Get historical posting frequency (30-37 days ago)
        $historicalPosts = $competitor->posts()
            ->whereBetween('published_at', [now()->subDays(37), now()->subDays(30)])
            ->count();

        if ($historicalPosts === 0) {
            return $alerts;
        }

        $changePercentage = (($recentPosts - $historicalPosts) / $historicalPosts) * 100;

        // Create alert if significant change (>50% increase or decrease)
        if (abs($changePercentage) > 50) {
            $severity = $this->determineSeverity(abs($changePercentage), [50, 100, 200]);
            $type = $changePercentage > 0 ? 'increased' : 'decreased';
            
            $alert = CompetitorAlert::create([
                'brand_id' => $competitor->brand_id,
                'competitor_brand_id' => $competitor->id,
                'tenant_id' => $competitor->tenant_id,
                'alert_type' => 'trend',
                'metric' => 'post_frequency',
                'severity' => $severity,
                'title' => "Posting Frequency Change Detected",
                'description' => "Competitor {$competitor->name} has {$type} their posting frequency by {$changePercentage}% in the past week.",
                'data' => [
                    'recent_posts' => $recentPosts,
                    'historical_posts' => $historicalPosts,
                    'change_percentage' => round($changePercentage, 2),
                ],
                'threshold_value' => $historicalPosts,
                'current_value' => $recentPosts,
                'change_percentage' => round($changePercentage, 2),
            ]);

            $alerts[] = $alert;
        }

        return $alerts;
    }

    /**
     * Check for viral posts.
     *
     * @param CompetitorBrand $competitor
     * @return array
     */
    protected function checkViralPosts(CompetitorBrand $competitor): array
    {
        $alerts = [];

        // Get posts from last 7 days
        $recentPosts = $competitor->posts()
            ->where('published_at', '>=', now()->subDays(7))
            ->get();

        if ($recentPosts->isEmpty()) {
            return $alerts;
        }

        // Calculate average engagement for competitor
        $avgEngagement = $competitor->posts()
            ->where('published_at', '>=', now()->subDays(30))
            ->avg('engagement_rate');

        // Check each recent post for viral performance (3x average)
        foreach ($recentPosts as $post) {
            if ($post->engagement_rate > ($avgEngagement * 3)) {
                // Check if alert already exists for this post
                $existingAlert = CompetitorAlert::where('competitor_post_id', $post->id)
                    ->where('alert_type', 'spike')
                    ->where('metric', 'viral_post')
                    ->first();

                if (!$existingAlert) {
                    $severity = $this->determineSeverity(
                        $post->engagement_rate / $avgEngagement,
                        [3, 5, 10]
                    );

                    $alert = CompetitorAlert::create([
                        'brand_id' => $competitor->brand_id,
                        'competitor_brand_id' => $competitor->id,
                        'tenant_id' => $competitor->tenant_id,
                        'competitor_post_id' => $post->id,
                        'alert_type' => 'spike',
                        'metric' => 'viral_post',
                        'severity' => $severity,
                        'title' => "Viral Post Detected",
                        'description' => "Competitor {$competitor->name} has a post performing {$post->engagement_rate}% engagement rate, significantly above their average of {$avgEngagement}%.",
                        'data' => [
                            'post_engagement_rate' => round($post->engagement_rate, 2),
                            'avg_engagement_rate' => round($avgEngagement, 2),
                            'multiplier' => round($post->engagement_rate / $avgEngagement, 2),
                            'post_caption' => substr($post->caption ?? '', 0, 100),
                            'post_permalink' => $post->permalink,
                        ],
                        'threshold_value' => round($avgEngagement, 2),
                        'current_value' => round($post->engagement_rate, 2),
                        'change_percentage' => round((($post->engagement_rate - $avgEngagement) / $avgEngagement) * 100, 2),
                    ]);

                    $alerts[] = $alert;
                }
            }
        }

        return $alerts;
    }

    /**
     * Determine alert severity based on thresholds.
     *
     * @param float $value
     * @param array $thresholds [low, medium, high]
     * @return string
     */
    protected function determineSeverity(float $value, array $thresholds): string
    {
        [$low, $medium, $high] = $thresholds;

        if ($value >= $high) {
            return 'critical';
        } elseif ($value >= $medium) {
            return 'high';
        } elseif ($value >= $low) {
            return 'medium';
        }

        return 'low';
    }

    /**
     * Get unread alerts for a brand.
     *
     * @param int $brandId
     * @param int $limit
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUnreadAlerts(int $brandId, int $limit = 10)
    {
        return CompetitorAlert::where('brand_id', $brandId)
            ->unread()
            ->active()
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->with(['competitorBrand', 'competitorPost'])
            ->get();
    }

    /**
     * Get alerts by severity for a brand.
     *
     * @param int $brandId
     * @param string $severity
     * @param int $limit
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAlertsBySeverity(int $brandId, string $severity, int $limit = 10)
    {
        return CompetitorAlert::where('brand_id', $brandId)
            ->severity($severity)
            ->active()
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->with(['competitorBrand', 'competitorPost'])
            ->get();
    }

    /**
     * Mark multiple alerts as read.
     *
     * @param array $alertIds
     * @return int
     */
    public function markAlertsAsRead(array $alertIds): int
    {
        return CompetitorAlert::whereIn('id', $alertIds)
            ->update([
                'is_read' => true,
                'read_at' => now(),
            ]);
    }

    /**
     * Dismiss multiple alerts.
     *
     * @param array $alertIds
     * @return int
     */
    public function dismissAlerts(array $alertIds): int
    {
        return CompetitorAlert::whereIn('id', $alertIds)
            ->update([
                'is_dismissed' => true,
                'dismissed_at' => now(),
            ]);
    }
}