<?php

namespace App\Services;

use App\Models\CampaignFinancial;
use App\Models\FinancialReport;
use App\Models\BudgetScenario;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AIFinancialAnalysisService
{
    private string $apiKey;
    private string $apiUrl;

    public function __construct()
    {
        $this->apiKey = config('services.openai.key');
        $this->apiUrl = config('services.openai.url', 'https://api.openai.com/v1/chat/completions');
    }

    /**
     * Analyze ROI and generate recommendations for a financial report
     */
    public function analyzeFinancialReport(FinancialReport $report): array
    {
        try {
            $report->markAsProcessing();

            // Gather financial data
            $financialData = $this->gatherFinancialData($report);

            // Generate AI analysis
            $analysis = $this->generateAIAnalysis($financialData);

            // Extract actionable insights
            $insights = $this->extractActionableInsights($analysis, $financialData);

            // Generate recommendations
            $recommendations = $this->generateRecommendations($analysis, $financialData);

            // Perform trend analysis
            $trendAnalysis = $this->analyzeTrends($financialData);

            $report->update([
                'recommendations_text' => $recommendations,
                'actionable_insights' => $insights,
                'trend_analysis' => $trendAnalysis,
            ]);

            $report->markAsCompleted();

            return [
                'success' => true,
                'recommendations' => $recommendations,
                'insights' => $insights,
                'trend_analysis' => $trendAnalysis,
            ];

        } catch (\Exception $e) {
            Log::error('AI Financial Analysis failed', [
                'report_id' => $report->id,
                'error' => $e->getMessage(),
            ]);

            $report->markAsFailed($e->getMessage());

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Generate budget scenarios with AI recommendations
     */
    public function generateBudgetScenarios(
        FinancialReport $report,
        float $totalBudget
    ): array {
        $financialData = $this->gatherFinancialData($report);

        $scenarios = [];

        // Conservative scenario
        $scenarios['conservative'] = $this->generateConservativeScenario(
            $report,
            $totalBudget,
            $financialData
        );

        // Balanced scenario
        $scenarios['balanced'] = $this->generateBalancedScenario(
            $report,
            $totalBudget,
            $financialData
        );

        // Aggressive scenario
        $scenarios['aggressive'] = $this->generateAggressiveScenario(
            $report,
            $totalBudget,
            $financialData
        );

        return $scenarios;
    }

    /**
     * Calculate ROI improvement potential
     */
    public function calculateROIImprovement(
        array $currentMetrics,
        array $proposedAllocation
    ): float {
        $currentROI = $currentMetrics['roi_percentage'] ?? 0;
        
        // Calculate weighted ROI based on channel performance
        $projectedROI = 0;
        foreach ($proposedAllocation as $channel => $percentage) {
            $channelROI = $currentMetrics['channel_roi'][$channel] ?? $currentROI;
            $projectedROI += ($channelROI * $percentage / 100);
        }

        return $projectedROI - $currentROI;
    }

    /**
     * Gather financial data for analysis
     */
    private function gatherFinancialData(FinancialReport $report): array
    {
        $data = [
            'report_id' => $report->id,
            'report_type' => $report->report_type,
            'total_spend' => $report->total_spend,
            'total_revenue' => $report->total_revenue,
            'net_profit' => $report->net_profit,
            'roi_percentage' => $report->roi_percentage,
            'total_conversions' => $report->total_conversions,
            'avg_conversion_rate' => $report->avg_conversion_rate,
            'avg_cost_per_conversion' => $report->avg_cost_per_conversion,
            'period_days' => $report->period_days,
        ];

        // Add spend breakdown
        if ($report->spend_breakdown) {
            $data['spend_breakdown'] = $report->spend_breakdown;
        }

        // Add revenue breakdown
        if ($report->revenue_breakdown) {
            $data['revenue_breakdown'] = $report->revenue_breakdown;
        }

        // Calculate channel performance if available
        $data['channel_roi'] = $this->calculateChannelROI($report);

        return $data;
    }

    /**
     * Calculate ROI per channel
     */
    private function calculateChannelROI(FinancialReport $report): array
    {
        $channelROI = [];

        if (!$report->spend_breakdown || !$report->revenue_breakdown) {
            return $channelROI;
        }

        foreach ($report->spend_breakdown as $channel => $spend) {
            $revenue = $report->revenue_breakdown[$channel] ?? 0;
            if ($spend > 0) {
                $channelROI[$channel] = (($revenue - $spend) / $spend) * 100;
            }
        }

        return $channelROI;
    }

    /**
     * Generate AI analysis using OpenAI
     */
    private function generateAIAnalysis(array $financialData): string
    {
        $prompt = $this->buildAnalysisPrompt($financialData);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->apiUrl, [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a financial analyst specializing in marketing ROI and campaign optimization. Provide actionable, data-driven insights.',
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt,
                    ],
                ],
                'temperature' => 0.7,
                'max_tokens' => 1000,
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'];
            }

            throw new \Exception('AI API request failed: ' . $response->body());

        } catch (\Exception $e) {
            Log::error('AI Analysis API call failed', ['error' => $e->getMessage()]);
            
            // Fallback to rule-based analysis
            return $this->generateRuleBasedAnalysis($financialData);
        }
    }

    /**
     * Build analysis prompt for AI
     */
    private function buildAnalysisPrompt(array $data): string
    {
        return "Analyze the following campaign financial data and provide insights:\n\n" .
               "Total Spend: $" . number_format($data['total_spend'], 2) . "\n" .
               "Total Revenue: $" . number_format($data['total_revenue'], 2) . "\n" .
               "Net Profit: $" . number_format($data['net_profit'], 2) . "\n" .
               "ROI: " . number_format($data['roi_percentage'], 2) . "%\n" .
               "Conversions: " . number_format($data['total_conversions']) . "\n" .
               "Conversion Rate: " . number_format($data['avg_conversion_rate'], 2) . "%\n" .
               "Cost per Conversion: $" . number_format($data['avg_cost_per_conversion'], 2) . "\n" .
               "Period: " . $data['period_days'] . " days\n\n" .
               ($data['channel_roi'] ? "Channel ROI:\n" . json_encode($data['channel_roi'], JSON_PRETTY_PRINT) . "\n\n" : "") .
               "Provide:\n" .
               "1. Overall performance assessment\n" .
               "2. At least 2 specific, actionable recommendations for improvement\n" .
               "3. Budget reallocation suggestions with expected ROI impact\n" .
               "4. Risk factors to consider";
    }

    /**
     * Generate rule-based analysis as fallback
     */
    private function generateRuleBasedAnalysis(array $data): string
    {
        $analysis = "Financial Performance Analysis:\n\n";

        // ROI Assessment
        if ($data['roi_percentage'] > 20) {
            $analysis .= "✓ Excellent ROI of " . number_format($data['roi_percentage'], 2) . "%. Campaign is highly profitable.\n";
        } elseif ($data['roi_percentage'] > 10) {
            $analysis .= "✓ Good ROI of " . number_format($data['roi_percentage'], 2) . "%. Campaign is performing well.\n";
        } elseif ($data['roi_percentage'] > 0) {
            $analysis .= "⚠ Positive but modest ROI of " . number_format($data['roi_percentage'], 2) . "%. Room for optimization.\n";
        } else {
            $analysis .= "⚠ Negative ROI of " . number_format($data['roi_percentage'], 2) . "%. Immediate optimization needed.\n";
        }

        // Conversion Rate Assessment
        if ($data['avg_conversion_rate'] < 2) {
            $analysis .= "⚠ Low conversion rate (" . number_format($data['avg_conversion_rate'], 2) . "%). Focus on improving landing pages and targeting.\n";
        } elseif ($data['avg_conversion_rate'] > 5) {
            $analysis .= "✓ Strong conversion rate (" . number_format($data['avg_conversion_rate'], 2) . "%). Maintain current strategies.\n";
        }

        // Channel Performance
        if (!empty($data['channel_roi'])) {
            $analysis .= "\nChannel Performance:\n";
            arsort($data['channel_roi']);
            foreach ($data['channel_roi'] as $channel => $roi) {
                $analysis .= "- " . ucfirst($channel) . ": " . number_format($roi, 2) . "% ROI\n";
            }
        }

        return $analysis;
    }

    /**
     * Extract actionable insights from analysis
     */
    private function extractActionableInsights(string $analysis, array $data): array
    {
        $insights = [];

        // Insight 1: Best performing channel
        if (!empty($data['channel_roi'])) {
            arsort($data['channel_roi']);
            $bestChannel = array_key_first($data['channel_roi']);
            $bestROI = $data['channel_roi'][$bestChannel];
            
            if ($bestROI > 15) {
                $insights[] = [
                    'type' => 'opportunity',
                    'priority' => 'high',
                    'title' => 'Scale High-Performing Channel',
                    'description' => "Shift 10-15% more budget to " . ucfirst($bestChannel) . " for potential +" . number_format($bestROI * 0.1, 1) . "% ROI improvement",
                    'expected_impact' => '+' . number_format($bestROI * 0.1, 1) . '% ROI',
                    'action' => 'increase_budget',
                    'channel' => $bestChannel,
                ];
            }
        }

        // Insight 2: Underperforming areas
        if ($data['avg_conversion_rate'] < 2) {
            $insights[] = [
                'type' => 'warning',
                'priority' => 'high',
                'title' => 'Improve Conversion Rate',
                'description' => 'Current conversion rate is below industry average. Optimize landing pages, improve targeting, and test new creatives',
                'expected_impact' => 'Potential 2-3x conversion improvement',
                'action' => 'optimize_conversion',
            ];
        }

        // Insight 3: Budget efficiency
        if ($data['roi_percentage'] > 0 && $data['roi_percentage'] < 15) {
            $insights[] = [
                'type' => 'recommendation',
                'priority' => 'medium',
                'title' => 'Optimize Budget Allocation',
                'description' => 'Reallocate budget from lower-performing channels to top performers for improved overall ROI',
                'expected_impact' => '+5-10% ROI improvement',
                'action' => 'reallocate_budget',
            ];
        }

        // Ensure at least 2 insights
        if (count($insights) < 2) {
            $insights[] = [
                'type' => 'recommendation',
                'priority' => 'medium',
                'title' => 'Test New Strategies',
                'description' => 'Allocate 10-15% of budget to test new channels, audiences, or creative approaches',
                'expected_impact' => 'Discover new growth opportunities',
                'action' => 'test_new_strategies',
            ];
        }

        return $insights;
    }

    /**
     * Generate recommendations text
     */
    private function generateRecommendations(string $analysis, array $data): string
    {
        $recommendations = "Based on the financial analysis:\n\n";
        
        $recommendations .= $analysis . "\n\n";
        
        $recommendations .= "Key Recommendations:\n";
        $recommendations .= "1. Monitor performance metrics daily and adjust budgets weekly\n";
        $recommendations .= "2. A/B test different creative approaches to improve conversion rates\n";
        $recommendations .= "3. Consider seasonal trends and adjust spending accordingly\n";
        $recommendations .= "4. Implement retargeting campaigns to improve ROI\n\n";
        
        $recommendations .= "Disclaimer: These are AI-generated estimates based on historical data. ";
        $recommendations .= "Actual results may vary. This is not financial advice.";

        return $recommendations;
    }

    /**
     * Analyze trends in financial data
     */
    private function analyzeTrends(array $data): array
    {
        return [
            'roi_trend' => $this->calculateTrend($data['roi_percentage']),
            'spend_efficiency' => $this->calculateSpendEfficiency($data),
            'conversion_trend' => $this->calculateTrend($data['avg_conversion_rate']),
            'revenue_growth' => $this->calculateRevenueGrowth($data),
        ];
    }

    /**
     * Calculate trend direction
     */
    private function calculateTrend(float $value): string
    {
        if ($value > 15) return 'strong_positive';
        if ($value > 5) return 'positive';
        if ($value > -5) return 'stable';
        return 'negative';
    }

    /**
     * Calculate spend efficiency
     */
    private function calculateSpendEfficiency(array $data): string
    {
        $efficiency = $data['total_revenue'] / max($data['total_spend'], 1);
        
        if ($efficiency > 2) return 'excellent';
        if ($efficiency > 1.5) return 'good';
        if ($efficiency > 1) return 'fair';
        return 'poor';
    }

    /**
     * Calculate revenue growth potential
     */
    private function calculateRevenueGrowth(array $data): array
    {
        return [
            'current' => $data['total_revenue'],
            'potential' => $data['total_revenue'] * 1.2, // 20% growth potential
            'growth_rate' => 20,
        ];
    }

    /**
     * Generate conservative budget scenario
     */
    private function generateConservativeScenario(
        FinancialReport $report,
        float $totalBudget,
        array $financialData
    ): BudgetScenario {
        $channelAllocation = $this->calculateConservativeAllocation($financialData);
        
        return BudgetScenario::create([
            'brand_id' => $report->brand_id,
            'financial_report_id' => $report->id,
            'name' => 'Conservative Budget Plan',
            'scenario_type' => 'conservative',
            'description' => 'Low-risk allocation focusing on proven channels',
            'total_budget' => $totalBudget,
            'channel_allocation' => $channelAllocation,
            'projected_revenue' => $totalBudget * 1.15, // 15% ROI
            'projected_roi' => 15,
            'projected_conversions' => (int)($totalBudget / ($financialData['avg_cost_per_conversion'] ?? 50)),
            'projected_cost_per_conversion' => $financialData['avg_cost_per_conversion'] ?? 50,
            'risk_assessment' => 'Low risk with stable, predictable returns',
            'status' => 'draft',
        ]);
    }

    /**
     * Generate balanced budget scenario
     */
    private function generateBalancedScenario(
        FinancialReport $report,
        float $totalBudget,
        array $financialData
    ): BudgetScenario {
        $channelAllocation = $this->calculateBalancedAllocation($financialData);
        
        return BudgetScenario::create([
            'brand_id' => $report->brand_id,
            'financial_report_id' => $report->id,
            'name' => 'Balanced Budget Plan',
            'scenario_type' => 'balanced',
            'description' => 'Moderate risk with mix of proven and growth channels',
            'total_budget' => $totalBudget,
            'channel_allocation' => $channelAllocation,
            'projected_revenue' => $totalBudget * 1.25, // 25% ROI
            'projected_roi' => 25,
            'projected_conversions' => (int)($totalBudget / (($financialData['avg_cost_per_conversion'] ?? 50) * 0.9)),
            'projected_cost_per_conversion' => ($financialData['avg_cost_per_conversion'] ?? 50) * 0.9,
            'risk_assessment' => 'Moderate risk with good growth potential',
            'status' => 'draft',
        ]);
    }

    /**
     * Generate aggressive budget scenario
     */
    private function generateAggressiveScenario(
        FinancialReport $report,
        float $totalBudget,
        array $financialData
    ): BudgetScenario {
        $channelAllocation = $this->calculateAggressiveAllocation($financialData);
        
        return BudgetScenario::create([
            'brand_id' => $report->brand_id,
            'financial_report_id' => $report->id,
            'name' => 'Aggressive Growth Plan',
            'scenario_type' => 'aggressive',
            'description' => 'High-risk, high-reward allocation for maximum growth',
            'total_budget' => $totalBudget,
            'channel_allocation' => $channelAllocation,
            'projected_revenue' => $totalBudget * 1.40, // 40% ROI
            'projected_roi' => 40,
            'projected_conversions' => (int)($totalBudget / (($financialData['avg_cost_per_conversion'] ?? 50) * 0.8)),
            'projected_cost_per_conversion' => ($financialData['avg_cost_per_conversion'] ?? 50) * 0.8,
            'risk_assessment' => 'Higher risk but significant growth potential',
            'status' => 'draft',
        ]);
    }

    /**
     * Calculate conservative channel allocation
     */
    private function calculateConservativeAllocation(array $data): array
    {
        // Focus on proven channels
        return [
            'instagram' => 40,
            'facebook' => 30,
            'google_ads' => 20,
            'other' => 10,
        ];
    }

    /**
     * Calculate balanced channel allocation
     */
    private function calculateBalancedAllocation(array $data): array
    {
        return [
            'instagram' => 35,
            'tiktok' => 25,
            'youtube' => 20,
            'facebook' => 15,
            'other' => 5,
        ];
    }

    /**
     * Calculate aggressive channel allocation
     */
    private function calculateAggressiveAllocation(array $data): array
    {
        // Focus on high-growth channels
        return [
            'tiktok' => 40,
            'instagram_reels' => 30,
            'youtube_shorts' => 20,
            'other' => 10,
        ];
    }
}