<?php

namespace App\Policies;

use App\Models\LocalizedContent;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class LocalizedContentPolicy
{
    use HandlesAuthorization;

    /**
     * Determine if the user can view any localized content.
     */
    public function viewAny(User $user): bool
    {
        return $user->role === 'brand_admin' || $user->role === 'agency';
    }

    /**
     * Determine if the user can view the localized content.
     */
    public function view(User $user, LocalizedContent $localizedContent): bool
    {
        // Brand admin can view their own brand's content
        if ($user->role === 'brand_admin') {
            return $user->brand_id === $localizedContent->brand_id;
        }

        // Agency can view if they have access to the brand
        if ($user->role === 'agency') {
            return $this->hasAgencyAccess($user, $localizedContent->brand_id);
        }

        return false;
    }

    /**
     * Determine if the user can create localized content.
     */
    public function create(User $user): bool
    {
        return $user->role === 'brand_admin' || $user->role === 'agency';
    }

    /**
     * Determine if the user can update the localized content.
     */
    public function update(User $user, LocalizedContent $localizedContent): bool
    {
        // Brand admin can update their own brand's content
        if ($user->role === 'brand_admin') {
            return $user->brand_id === $localizedContent->brand_id;
        }

        // Approved agency can update
        if ($user->role === 'agency') {
            return $this->hasAgencyAccess($user, $localizedContent->brand_id);
        }

        return false;
    }

    /**
     * Determine if the user can approve/reject the localized content.
     */
    public function approve(User $user, LocalizedContent $localizedContent): bool
    {
        // Only brand admin can approve/reject
        if ($user->role === 'brand_admin') {
            return $user->brand_id === $localizedContent->brand_id;
        }

        // Approved agency with proper permissions can also approve
        if ($user->role === 'agency') {
            return $this->hasAgencyApprovalAccess($user, $localizedContent->brand_id);
        }

        return false;
    }

    /**
     * Determine if the user can delete the localized content.
     */
    public function delete(User $user, LocalizedContent $localizedContent): bool
    {
        // Only brand admin can delete
        if ($user->role === 'brand_admin') {
            return $user->brand_id === $localizedContent->brand_id;
        }

        return false;
    }

    /**
     * Determine if the user can manage localization for a brand.
     */
    public function manageLocalization(User $user, $brand): bool
    {
        $brandId = is_object($brand) ? $brand->id : $brand;

        // Brand admin can manage their own brand
        if ($user->role === 'brand_admin') {
            return $user->brand_id === $brandId;
        }

        // Approved agency can manage
        if ($user->role === 'agency') {
            return $this->hasAgencyAccess($user, $brandId);
        }

        return false;
    }

    /**
     * Check if agency has access to the brand.
     */
    protected function hasAgencyAccess(User $user, int $brandId): bool
    {
        // Check if there's an active relationship between agency and brand
        // This assumes you have a pivot table or relationship model
        return \DB::table('brand_agency')
            ->where('agency_id', $user->id)
            ->where('brand_id', $brandId)
            ->where('status', 'approved')
            ->exists();
    }

    /**
     * Check if agency has approval access to the brand.
     */
    protected function hasAgencyApprovalAccess(User $user, int $brandId): bool
    {
        // Check if agency has approval permissions
        return \DB::table('brand_agency')
            ->where('agency_id', $user->id)
            ->where('brand_id', $brandId)
            ->where('status', 'approved')
            ->where('can_approve_content', true)
            ->exists();
    }
}