<?php

namespace App\Policies;

use App\Models\User;
use App\Models\CreatorSocialConnection;
use App\Models\SharedSocialAccess;

class CreatorSocialConnectionPolicy
{
    /**
     * Determine if the user can view any connections.
     */
    public function viewAny(User $user): bool
    {
        return $user->role === 'creator' || $user->role === 'superadmin';
    }

    /**
     * Determine if the user can view the connection.
     */
    public function view(User $user, CreatorSocialConnection $connection): bool
    {
        // SuperAdmin can view all
        if ($user->role === 'superadmin') {
            return true;
        }

        // Creator can only view their own connections
        return $user->id === $connection->creator_id;
    }

    /**
     * Determine if the user can create connections.
     */
    public function create(User $user): bool
    {
        return $user->role === 'creator';
    }

    /**
     * Determine if the user can update the connection.
     */
    public function update(User $user, CreatorSocialConnection $connection): bool
    {
        // SuperAdmin can update all
        if ($user->role === 'superadmin') {
            return true;
        }

        // Creator can only update their own connections
        return $user->id === $connection->creator_id;
    }

    /**
     * Determine if the user can delete the connection.
     */
    public function delete(User $user, CreatorSocialConnection $connection): bool
    {
        // SuperAdmin can delete all
        if ($user->role === 'superadmin') {
            return true;
        }

        // Creator can only delete their own connections
        return $user->id === $connection->creator_id;
    }

    /**
     * Determine if the user can share the connection with brands.
     */
    public function share(User $user, CreatorSocialConnection $connection): bool
    {
        // Only the creator who owns the connection can share it
        return $user->id === $connection->creator_id && $connection->canBeShared();
    }

    /**
     * Determine if the user can revoke shared access.
     */
    public function revoke(User $user, SharedSocialAccess $sharedAccess): bool
    {
        // SuperAdmin can revoke any access
        if ($user->role === 'superadmin') {
            return true;
        }

        // Creator can revoke their own shared access
        if ($user->id === $sharedAccess->creator_id) {
            return true;
        }

        // Brand admin can revoke access to their brand
        if ($user->role === 'brand_admin' && $user->brand_id === $sharedAccess->brand_id) {
            return true;
        }

        return false;
    }

    /**
     * Determine if the user can approve ad usage.
     */
    public function approveAdUsage(User $user, CreatorSocialConnection $connection): bool
    {
        // Only the creator who owns the connection can approve ad usage
        return $user->id === $connection->creator_id;
    }

    /**
     * Determine if the user can refresh tokens.
     */
    public function refreshToken(User $user, CreatorSocialConnection $connection): bool
    {
        // SuperAdmin can refresh any token
        if ($user->role === 'superadmin') {
            return true;
        }

        // Creator can refresh their own tokens
        return $user->id === $connection->creator_id;
    }
}