<?php

namespace App\Notifications;

use App\Models\SharedSocialAccess;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SocialAccessRevokedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $sharedAccess;
    protected $recipientType;

    /**
     * Create a new notification instance.
     */
    public function __construct(SharedSocialAccess $sharedAccess, string $recipientType)
    {
        $this->sharedAccess = $sharedAccess;
        $this->recipientType = $recipientType; // 'creator' or 'brand'
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable): MailMessage
    {
        $connection = $this->sharedAccess->connection;
        $brand = $this->sharedAccess->brand;
        $creator = $this->sharedAccess->creator;
        $reason = $this->sharedAccess->revocation_reason;

        if ($this->recipientType === 'creator') {
            $mail = (new MailMessage)
                ->subject('Social Account Access Revoked')
                ->greeting("Hello {$creator->name}!")
                ->line("Access to your {$connection->platform_display_name} account has been revoked.")
                ->line("**Brand:** {$brand->name}")
                ->line("**Platform:** {$connection->platform_display_name}")
                ->line("**Revoked:** {$this->sharedAccess->revoked_at->format('M d, Y H:i')}")
                ->action('View Connection Details', route('creator.social-connections.show', $connection));

            if ($reason) {
                $mail->line("**Reason:** {$reason}");
            }

            return $mail->line('The brand no longer has access to your account.')
                ->line('Thank you for using our platform!');
        } else {
            $mail = (new MailMessage)
                ->subject('Creator Revoked Social Account Access')
                ->greeting("Hello!")
                ->line("A creator has revoked your brand's access to their {$connection->platform_display_name} account.")
                ->line("**Creator:** {$creator->name}")
                ->line("**Platform:** {$connection->platform_display_name}")
                ->line("**Account:** @{$connection->platform_username}")
                ->line("**Revoked:** {$this->sharedAccess->revoked_at->format('M d, Y H:i')}")
                ->action('View Creator Profile', route('brand.creators.show', $creator));

            if ($reason) {
                $mail->line("**Reason:** {$reason}");
            }

            return $mail->line('You no longer have access to this creator\'s social account.')
                ->line('Thank you for using our platform!');
        }
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray($notifiable): array
    {
        $connection = $this->sharedAccess->connection;
        $brand = $this->sharedAccess->brand;
        $creator = $this->sharedAccess->creator;

        if ($this->recipientType === 'creator') {
            return [
                'type' => 'social_access_revoked',
                'title' => 'Social Account Access Revoked',
                'message' => "You revoked {$brand->name}'s access to your {$connection->platform_display_name} account",
                'connection_id' => $connection->id,
                'brand_id' => $brand->id,
                'shared_access_id' => $this->sharedAccess->id,
                'platform' => $connection->platform,
                'revoked_at' => $this->sharedAccess->revoked_at->toIso8601String(),
                'reason' => $this->sharedAccess->revocation_reason,
                'action_url' => route('creator.social-connections.show', $connection),
            ];
        } else {
            return [
                'type' => 'social_access_lost',
                'title' => 'Creator Revoked Social Access',
                'message' => "{$creator->name} revoked access to their {$connection->platform_display_name} account",
                'connection_id' => $connection->id,
                'creator_id' => $creator->id,
                'shared_access_id' => $this->sharedAccess->id,
                'platform' => $connection->platform,
                'revoked_at' => $this->sharedAccess->revoked_at->toIso8601String(),
                'reason' => $this->sharedAccess->revocation_reason,
                'action_url' => route('brand.creators.show', $creator),
            ];
        }
    }
}