<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ReelPosted extends Notification implements ShouldQueue
{
    use Queueable;

    protected $scheduledReel;
    protected $successCount;
    protected $failureCount;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct($scheduledReel, $successCount, $failureCount)
    {
        $this->scheduledReel = $scheduledReel;
        $this->successCount = $successCount;
        $this->failureCount = $failureCount;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $message = (new MailMessage)
            ->subject('Reel Posting Status')
            ->line('Your scheduled reel has been processed.')
            ->line('Template: ' . $this->scheduledReel->template);

        if ($this->successCount > 0) {
            $message->line("Successfully posted to {$this->successCount} platform(s).");
        }

        if ($this->failureCount > 0) {
            $message->line("Failed to post to {$this->failureCount} platform(s).");
        }

        $message->action('View Reel', url('/' . ($this->scheduledReel->owner_type === 'brand' ? 'brand' : 'creator') . '/reels/' . $this->scheduledReel->id))
            ->line('Thank you for using our application!');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'message' => 'Your scheduled reel has been processed.',
            'reel_id' => $this->scheduledReel->id,
            'template' => $this->scheduledReel->template,
            'success_count' => $this->successCount,
            'failure_count' => $this->failureCount,
            'status' => $this->scheduledReel->status,
        ];
    }
}