<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SupportMessage extends Model
{
    use HasFactory;

    protected $fillable = [
        'conversation_id',
        'sender_type',
        'sender_id',
        'message_text',
        'message_text_redacted',
        'detected_pii',
        'bot_response',
        'citations',
        'confidence_score',
        'requires_escalation',
        'escalation_reason',
        'metadata',
        'helpful_flag',
        'feedback_comment',
    ];

    protected $casts = [
        'detected_pii' => 'array',
        'citations' => 'array',
        'confidence_score' => 'decimal:2',
        'requires_escalation' => 'boolean',
        'metadata' => 'array',
        'helpful_flag' => 'boolean',
    ];

    /**
     * Get the conversation this message belongs to.
     */
    public function conversation()
    {
        return $this->belongsTo(SupportConversation::class, 'conversation_id');
    }

    /**
     * Get the sender (if user).
     */
    public function sender()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    /**
     * Scope to get customer messages.
     */
    public function scopeFromCustomer($query)
    {
        return $query->where('sender_type', 'customer');
    }

    /**
     * Scope to get bot messages.
     */
    public function scopeFromBot($query)
    {
        return $query->where('sender_type', 'bot');
    }

    /**
     * Scope to get agent messages.
     */
    public function scopeFromAgent($query)
    {
        return $query->where('sender_type', 'agent');
    }

    /**
     * Scope to get low confidence messages.
     */
    public function scopeLowConfidence($query, float $threshold = 70.0)
    {
        return $query->where('confidence_score', '<', $threshold);
    }

    /**
     * Check if message is from customer.
     */
    public function isFromCustomer(): bool
    {
        return $this->sender_type === 'customer';
    }

    /**
     * Check if message is from bot.
     */
    public function isFromBot(): bool
    {
        return $this->sender_type === 'bot';
    }

    /**
     * Check if message is from agent.
     */
    public function isFromAgent(): bool
    {
        return $this->sender_type === 'agent';
    }

    /**
     * Check if message has PII.
     */
    public function hasPII(): bool
    {
        return !empty($this->detected_pii);
    }

    /**
     * Get safe message text (redacted if PII detected).
     */
    public function getSafeMessageTextAttribute(): string
    {
        return $this->message_text_redacted ?? $this->message_text;
    }

    /**
     * Mark message as helpful.
     */
    public function markHelpful(?string $comment = null): void
    {
        $this->update([
            'helpful_flag' => true,
            'feedback_comment' => $comment,
        ]);
    }

    /**
     * Mark message as not helpful.
     */
    public function markNotHelpful(?string $comment = null): void
    {
        $this->update([
            'helpful_flag' => false,
            'feedback_comment' => $comment,
        ]);
    }

    /**
     * Get confidence level as string.
     */
    public function getConfidenceLevelAttribute(): string
    {
        if (!$this->confidence_score) {
            return 'unknown';
        }

        if ($this->confidence_score >= 90) {
            return 'very_high';
        } elseif ($this->confidence_score >= 75) {
            return 'high';
        } elseif ($this->confidence_score >= 60) {
            return 'medium';
        } else {
            return 'low';
        }
    }
}