<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SupportBotSettings extends Model
{
    use HasFactory;

    protected $fillable = [
        'brand_id',
        'bot_enabled',
        'confidence_threshold',
        'max_messages_before_escalation',
        'business_hours',
        'welcome_message',
        'escalation_message',
        'offline_message',
        'widget_settings',
        'pii_redaction_rules',
        'collect_feedback',
        'enable_human_handoff',
        'notification_settings',
        'fallback_email',
    ];

    protected $casts = [
        'bot_enabled' => 'boolean',
        'confidence_threshold' => 'decimal:2',
        'business_hours' => 'array',
        'widget_settings' => 'array',
        'pii_redaction_rules' => 'array',
        'collect_feedback' => 'boolean',
        'enable_human_handoff' => 'boolean',
        'notification_settings' => 'array',
    ];

    /**
     * Get the brand that owns the settings.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get default welcome message.
     */
    public static function defaultWelcomeMessage(): string
    {
        return "Hello! I'm here to help you. How can I assist you today?";
    }

    /**
     * Get default escalation message.
     */
    public static function defaultEscalationMessage(): string
    {
        return "I'm connecting you with a human agent who can better assist you. Please hold on.";
    }

    /**
     * Get default offline message.
     */
    public static function defaultOfflineMessage(): string
    {
        return "We're currently offline. Please leave your message and we'll get back to you soon.";
    }

    /**
     * Check if bot is currently available based on business hours.
     */
    public function isBotAvailable(): bool
    {
        if (!$this->bot_enabled) {
            return false;
        }

        if (empty($this->business_hours)) {
            return true; // 24/7 if no hours specified
        }

        $now = now();
        $dayOfWeek = strtolower($now->format('l'));
        
        if (!isset($this->business_hours[$dayOfWeek])) {
            return false;
        }

        $hours = $this->business_hours[$dayOfWeek];
        
        if ($hours === 'closed') {
            return false;
        }

        if ($hours === '24/7') {
            return true;
        }

        // Parse hours like "09:00-17:00"
        if (isset($hours['start']) && isset($hours['end'])) {
            $currentTime = $now->format('H:i');
            return $currentTime >= $hours['start'] && $currentTime <= $hours['end'];
        }

        return true;
    }

    /**
     * Get widget settings with defaults.
     */
    public function getWidgetSettingsWithDefaults(): array
    {
        $defaults = [
            'position' => 'bottom-right',
            'primary_color' => '#007bff',
            'text_color' => '#ffffff',
            'show_branding' => true,
            'avatar_url' => null,
            'greeting_delay' => 2000, // ms
        ];

        return array_merge($defaults, $this->widget_settings ?? []);
    }

    /**
     * Get notification settings with defaults.
     */
    public function getNotificationSettingsWithDefaults(): array
    {
        $defaults = [
            'email_on_escalation' => true,
            'email_on_low_confidence' => false,
            'slack_webhook' => null,
            'notify_on_negative_feedback' => true,
        ];

        return array_merge($defaults, $this->notification_settings ?? []);
    }

    /**
     * Should escalate based on confidence score.
     */
    public function shouldEscalate(float $confidenceScore): bool
    {
        return $confidenceScore < $this->confidence_threshold;
    }

    /**
     * Should escalate based on message count.
     */
    public function shouldEscalateByMessageCount(int $messageCount): bool
    {
        return $messageCount >= $this->max_messages_before_escalation;
    }
}