<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SubscriptionInvoice extends Model
{
    protected $fillable = [
        'subscription_id',
        'user_id',
        'transaction_id',
        'invoice_number',
        'subtotal',
        'tax',
        'total',
        'currency',
        'status',
        'issued_at',
        'due_at',
        'paid_at',
        'line_items',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax' => 'decimal:2',
        'total' => 'decimal:2',
        'line_items' => 'array',
        'issued_at' => 'datetime',
        'due_at' => 'datetime',
        'paid_at' => 'datetime',
    ];

    // Invoice statuses
    const STATUS_DRAFT = 'draft';
    const STATUS_PAID = 'paid';
    const STATUS_VOID = 'void';
    const STATUS_REFUNDED = 'refunded';

    /**
     * Get the subscription that owns the invoice.
     */
    public function subscription(): BelongsTo
    {
        return $this->belongsTo(Subscription::class);
    }

    /**
     * Get the user that owns the invoice.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the transaction for this invoice.
     */
    public function transaction(): BelongsTo
    {
        return $this->belongsTo(SubscriptionTransaction::class, 'transaction_id');
    }

    /**
     * Check if invoice is paid.
     */
    public function isPaid(): bool
    {
        return $this->status === self::STATUS_PAID;
    }

    /**
     * Check if invoice is overdue.
     */
    public function isOverdue(): bool
    {
        return !$this->isPaid() && 
               $this->due_at && 
               $this->due_at->isPast();
    }

    /**
     * Mark invoice as paid.
     */
    public function markAsPaid(): bool
    {
        $this->status = self::STATUS_PAID;
        $this->paid_at = now();
        return $this->save();
    }

    /**
     * Mark invoice as void.
     */
    public function markAsVoid(): bool
    {
        $this->status = self::STATUS_VOID;
        return $this->save();
    }

    /**
     * Get formatted total.
     */
    public function getFormattedTotalAttribute(): string
    {
        $symbol = $this->currency === 'INR' ? '₹' : '$';
        return $symbol . number_format($this->total, 2);
    }

    /**
     * Get formatted subtotal.
     */
    public function getFormattedSubtotalAttribute(): string
    {
        $symbol = $this->currency === 'INR' ? '₹' : '$';
        return $symbol . number_format($this->subtotal, 2);
    }

    /**
     * Get formatted tax.
     */
    public function getFormattedTaxAttribute(): string
    {
        $symbol = $this->currency === 'INR' ? '₹' : '$';
        return $symbol . number_format($this->tax, 2);
    }

    /**
     * Generate a unique invoice number.
     */
    public static function generateInvoiceNumber(): string
    {
        $prefix = 'INV';
        $date = now()->format('Ymd');
        $random = strtoupper(substr(md5(uniqid()), 0, 6));
        
        return "{$prefix}-{$date}-{$random}";
    }

    /**
     * Scope to get paid invoices.
     */
    public function scopePaid($query)
    {
        return $query->where('status', self::STATUS_PAID);
    }

    /**
     * Scope to get overdue invoices.
     */
    public function scopeOverdue($query)
    {
        return $query->where('status', '!=', self::STATUS_PAID)
                     ->where('due_at', '<', now());
    }
}