<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SocialComment extends Model
{
    use HasFactory;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'social_post_id',
        'social_account_id',
        'brand_id',
        'creator_id',
        'provider_comment_id',
        'content',
        'author_name',
        'author_username',
        'author_avatar_url',
        'is_reply',
        'parent_id',
        'raw_data',
        'published_at',
        'last_checked_at',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'raw_data' => 'array',
        'published_at' => 'datetime',
        'last_checked_at' => 'datetime',
        'is_reply' => 'boolean',
    ];
    
    /**
     * Get the social post that owns the comment.
     */
    public function socialPost()
    {
        return $this->belongsTo(SocialPost::class);
    }
    
    /**
     * Get the social account that owns the comment.
     */
    public function socialAccount()
    {
        return $this->belongsTo(SocialAccount::class);
    }
    
    /**
     * Get the brand that owns the comment.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }
    
    /**
     * Get the creator that owns the comment.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'creator_id');
    }
    
    /**
     * Get the parent comment.
     */
    public function parent()
    {
        return $this->belongsTo(SocialComment::class, 'parent_id');
    }
    
    /**
     * Get the child comments.
     */
    public function children()
    {
        return $this->hasMany(SocialComment::class, 'parent_id');
    }
    
    /**
     * Get the auto-reply suggestions for this comment.
     */
    public function autoReplySuggestions()
    {
        return $this->hasMany(AutoReplySuggestion::class, 'social_comment_id');
    }
    
    /**
     * Get the latest auto-reply suggestion for this comment.
     */
    public function latestAutoReplySuggestion()
    {
        return $this->hasOne(AutoReplySuggestion::class, 'social_comment_id')->latest();
    }
}