<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ReputationDriver extends Model
{
    use HasFactory;

    protected $fillable = [
        'reputation_score_id',
        'brand_id',
        'type',
        'category',
        'source_type',
        'source_id',
        'source_url',
        'source_platform',
        'source_content',
        'impact_score',
        'engagement_count',
        'sentiment_value',
        'occurred_at',
        'metadata',
        'is_resolved',
        'resolution_note',
        'resolved_by',
        'resolved_at',
    ];

    protected $casts = [
        'impact_score' => 'decimal:2',
        'engagement_count' => 'integer',
        'sentiment_value' => 'decimal:2',
        'occurred_at' => 'datetime',
        'metadata' => 'array',
        'is_resolved' => 'boolean',
        'resolved_at' => 'datetime',
    ];

    /**
     * Get the reputation score that owns the driver.
     */
    public function reputationScore(): BelongsTo
    {
        return $this->belongsTo(ReputationScore::class);
    }

    /**
     * Get the brand that owns the driver.
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the user who resolved this driver.
     */
    public function resolver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'resolved_by');
    }

    /**
     * Mark this driver as resolved.
     */
    public function markAsResolved(User $user, string $note = null): void
    {
        $this->update([
            'is_resolved' => true,
            'resolution_note' => $note,
            'resolved_by' => $user->id,
            'resolved_at' => now(),
        ]);
    }

    /**
     * Get the impact level (high, medium, low).
     */
    public function getImpactLevelAttribute(): string
    {
        $impact = abs($this->impact_score);
        
        if ($impact >= 5) return 'high';
        if ($impact >= 2) return 'medium';
        return 'low';
    }

    /**
     * Get the sentiment label.
     */
    public function getSentimentLabelAttribute(): ?string
    {
        if ($this->sentiment_value === null) {
            return null;
        }

        if ($this->sentiment_value >= 0.5) return 'very_positive';
        if ($this->sentiment_value >= 0.1) return 'positive';
        if ($this->sentiment_value >= -0.1) return 'neutral';
        if ($this->sentiment_value >= -0.5) return 'negative';
        return 'very_negative';
    }

    /**
     * Get formatted source content (truncated).
     */
    public function getExcerptAttribute(): string
    {
        if (!$this->source_content) {
            return '';
        }

        return strlen($this->source_content) > 200
            ? substr($this->source_content, 0, 200) . '...'
            : $this->source_content;
    }

    /**
     * Scope to get positive drivers.
     */
    public function scopePositive($query)
    {
        return $query->where('type', 'positive');
    }

    /**
     * Scope to get negative drivers.
     */
    public function scopeNegative($query)
    {
        return $query->where('type', 'negative');
    }

    /**
     * Scope to get unresolved drivers.
     */
    public function scopeUnresolved($query)
    {
        return $query->where('is_resolved', false);
    }

    /**
     * Scope to get resolved drivers.
     */
    public function scopeResolved($query)
    {
        return $query->where('is_resolved', true);
    }

    /**
     * Scope to get drivers by category.
     */
    public function scopeByCategory($query, string $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Scope to get high impact drivers.
     */
    public function scopeHighImpact($query)
    {
        return $query->where('impact_score', '>=', 5)
            ->orWhere('impact_score', '<=', -5);
    }

    /**
     * Scope to order by impact.
     */
    public function scopeOrderByImpact($query, string $direction = 'desc')
    {
        return $query->orderBy('impact_score', $direction);
    }
}