<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class FinancialReport extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'campaign_id',
        'brand_id',
        'report_type',
        'total_spend',
        'total_revenue',
        'net_profit',
        'roi_percentage',
        'breakeven_days',
        'total_clicks',
        'total_impressions',
        'total_conversions',
        'avg_conversion_rate',
        'avg_cost_per_conversion',
        'recommendations_text',
        'actionable_insights',
        'spend_breakdown',
        'revenue_breakdown',
        'trend_analysis',
        'period_start',
        'period_end',
        'ai_status',
        'ai_processed_at',
        'ai_error',
        'export_count',
        'last_exported_at',
    ];

    protected $casts = [
        'total_spend' => 'decimal:2',
        'total_revenue' => 'decimal:2',
        'net_profit' => 'decimal:2',
        'roi_percentage' => 'decimal:2',
        'breakeven_days' => 'integer',
        'total_clicks' => 'integer',
        'total_impressions' => 'integer',
        'total_conversions' => 'integer',
        'avg_conversion_rate' => 'decimal:2',
        'avg_cost_per_conversion' => 'decimal:2',
        'actionable_insights' => 'array',
        'spend_breakdown' => 'array',
        'revenue_breakdown' => 'array',
        'trend_analysis' => 'array',
        'period_start' => 'date',
        'period_end' => 'date',
        'ai_processed_at' => 'datetime',
        'last_exported_at' => 'datetime',
        'export_count' => 'integer',
    ];

    /**
     * Get the campaign that owns the report
     */
    public function campaign(): BelongsTo
    {
        return $this->belongsTo(Campaign::class);
    }

    /**
     * Get the brand that owns the report
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(User::class, 'brand_id');
    }

    /**
     * Get the budget scenarios for this report
     */
    public function budgetScenarios(): HasMany
    {
        return $this->hasMany(BudgetScenario::class);
    }

    /**
     * Scope to filter by report type
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('report_type', $type);
    }

    /**
     * Scope to filter by brand
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope to filter by AI processing status
     */
    public function scopeWithAiStatus($query, string $status)
    {
        return $query->where('ai_status', $status);
    }

    /**
     * Scope to get recent reports
     */
    public function scopeRecent($query, int $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Mark report as AI processing
     */
    public function markAsProcessing(): void
    {
        $this->update([
            'ai_status' => 'processing',
        ]);
    }

    /**
     * Mark report as AI completed
     */
    public function markAsCompleted(): void
    {
        $this->update([
            'ai_status' => 'completed',
            'ai_processed_at' => now(),
            'ai_error' => null,
        ]);
    }

    /**
     * Mark report as AI failed
     */
    public function markAsFailed(string $error): void
    {
        $this->update([
            'ai_status' => 'failed',
            'ai_error' => $error,
        ]);
    }

    /**
     * Increment export count
     */
    public function incrementExportCount(): void
    {
        $this->increment('export_count');
        $this->update(['last_exported_at' => now()]);
    }

    /**
     * Check if report is profitable
     */
    public function isProfitable(): bool
    {
        return $this->net_profit > 0;
    }

    /**
     * Get ROI status
     */
    public function getRoiStatus(): string
    {
        if ($this->roi_percentage === null) {
            return 'unknown';
        }

        if ($this->roi_percentage > 20) {
            return 'excellent';
        } elseif ($this->roi_percentage > 10) {
            return 'good';
        } elseif ($this->roi_percentage > 0) {
            return 'positive';
        } elseif ($this->roi_percentage < -10) {
            return 'poor';
        }

        return 'breakeven';
    }

    /**
     * Get number of actionable insights
     */
    public function getInsightsCountAttribute(): int
    {
        return is_array($this->actionable_insights) ? count($this->actionable_insights) : 0;
    }

    /**
     * Check if AI processing is complete
     */
    public function isAiProcessed(): bool
    {
        return $this->ai_status === 'completed';
    }

    /**
     * Check if AI processing failed
     */
    public function hasAiFailed(): bool
    {
        return $this->ai_status === 'failed';
    }

    /**
     * Get period duration in days
     */
    public function getPeriodDaysAttribute(): int
    {
        return $this->period_start->diffInDays($this->period_end);
    }
}