<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CompetitorInsight extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'brand_id',
        'competitor_brand_id',
        'tenant_id',
        'start_date',
        'end_date',
        'competitor_metrics',
        'brand_metrics',
        'comparison_data',
        'topics',
        'sentiment_trends',
        'top_posts',
        'engagement_patterns',
        'content_strategy',
        'hashtag_analysis',
        'recommendations',
        'summary',
        'status',
        'error_message',
        'generated_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'competitor_metrics' => 'array',
        'brand_metrics' => 'array',
        'comparison_data' => 'array',
        'topics' => 'array',
        'sentiment_trends' => 'array',
        'top_posts' => 'array',
        'engagement_patterns' => 'array',
        'content_strategy' => 'array',
        'hashtag_analysis' => 'array',
        'recommendations' => 'array',
        'generated_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the insight.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the competitor brand for this insight.
     */
    public function competitorBrand()
    {
        return $this->belongsTo(CompetitorBrand::class);
    }

    /**
     * Get the tenant that owns the insight.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Check if the insight is completed.
     *
     * @return bool
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Check if the insight is pending.
     *
     * @return bool
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Check if the insight is processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return $this->status === 'processing';
    }

    /**
     * Check if the insight has failed.
     *
     * @return bool
     */
    public function hasFailed()
    {
        return $this->status === 'failed';
    }

    /**
     * Mark the insight as processing.
     *
     * @return $this
     */
    public function markAsProcessing()
    {
        $this->update(['status' => 'processing']);
        return $this;
    }

    /**
     * Mark the insight as completed.
     *
     * @return $this
     */
    public function markAsCompleted()
    {
        $this->update([
            'status' => 'completed',
            'generated_at' => now(),
        ]);
        return $this;
    }

    /**
     * Mark the insight as failed.
     *
     * @param string $errorMessage
     * @return $this
     */
    public function markAsFailed(string $errorMessage)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
        ]);
        return $this;
    }

    /**
     * Get a specific metric comparison.
     *
     * @param string $metric
     * @return array|null
     */
    public function getMetricComparison(string $metric)
    {
        return $this->comparison_data[$metric] ?? null;
    }

    /**
     * Get the performance difference percentage for a metric.
     *
     * @param string $metric
     * @return float|null
     */
    public function getPerformanceDifference(string $metric)
    {
        $comparison = $this->getMetricComparison($metric);
        return $comparison['difference_percentage'] ?? null;
    }

    /**
     * Scope a query to only include completed insights.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope a query to filter by brand.
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope a query to filter by competitor.
     */
    public function scopeForCompetitor($query, $competitorBrandId)
    {
        return $query->where('competitor_brand_id', $competitorBrandId);
    }

    /**
     * Scope a query to filter by date range.
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->where('start_date', '>=', $startDate)
                     ->where('end_date', '<=', $endDate);
    }

    /**
     * Scope a query to get the latest insight.
     */
    public function scopeLatest($query)
    {
        return $query->orderBy('created_at', 'desc');
    }
}