<?php

namespace App\Jobs;

use App\Models\PerformanceForecast;
use App\Services\PerformanceForecastService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessPerformanceForecast implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The performance forecast instance.
     *
     * @var \App\Models\PerformanceForecast
     */
    public $forecast;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 300; // 5 minutes

    /**
     * Create a new job instance.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return void
     */
    public function __construct(PerformanceForecast $forecast)
    {
        $this->forecast = $forecast;
    }

    /**
     * Execute the job.
     *
     * @param  \App\Services\PerformanceForecastService  $forecastService
     * @return void
     */
    public function handle(PerformanceForecastService $forecastService)
    {
        Log::info('Processing performance forecast', [
            'forecast_id' => $this->forecast->id,
            'brand_id' => $this->forecast->brand_id,
        ]);

        try {
            // Process the forecast
            $forecastService->processForecast($this->forecast);

            Log::info('Performance forecast processed successfully', [
                'forecast_id' => $this->forecast->id,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to process performance forecast', [
                'forecast_id' => $this->forecast->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            // Mark forecast as failed
            $this->forecast->markAsFailed($e->getMessage());

            // Re-throw the exception to trigger retry logic
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     *
     * @param  \Throwable  $exception
     * @return void
     */
    public function failed(\Throwable $exception)
    {
        Log::error('Performance forecast job failed permanently', [
            'forecast_id' => $this->forecast->id,
            'error' => $exception->getMessage(),
        ]);

        // Mark forecast as failed if not already marked
        if (!$this->forecast->hasFailed()) {
            $this->forecast->markAsFailed(
                'Job failed after ' . $this->tries . ' attempts: ' . $exception->getMessage()
            );
        }
    }

    /**
     * Get the tags that should be assigned to the job.
     *
     * @return array
     */
    public function tags()
    {
        return [
            'forecast',
            'forecast:' . $this->forecast->id,
            'brand:' . $this->forecast->brand_id,
        ];
    }
}