<?php

namespace App\Http\Controllers;

use App\Models\SupportConversation;
use App\Models\SupportMessage;
use App\Models\SupportBotSettings;
use App\Models\Brand;
use App\Services\RAGService;
use App\Services\PIIRedactionService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;

class SupportBotController extends Controller
{
    protected RAGService $ragService;
    protected PIIRedactionService $piiService;

    public function __construct(RAGService $ragService, PIIRedactionService $piiService)
    {
        $this->ragService = $ragService;
        $this->piiService = $piiService;
    }

    /**
     * Start a new support conversation.
     */
    public function startConversation(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'customer_name' => 'nullable|string|max:255',
            'customer_email' => 'nullable|email|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'initial_query' => 'nullable|string|max:1000',
            'channel' => 'nullable|in:widget,email,api,chat',
            'metadata' => 'nullable|array',
        ]);

        try {
            $brand = Brand::findOrFail($validated['brand_id']);
            $settings = SupportBotSettings::where('brand_id', $brand->id)->first();

            // Check if bot is available
            if (!$settings || !$settings->isBotAvailable()) {
                return response()->json([
                    'success' => false,
                    'message' => $settings?->offline_message ?? SupportBotSettings::defaultOfflineMessage(),
                    'bot_available' => false,
                ], 200);
            }

            $conversation = SupportConversation::create([
                'brand_id' => $validated['brand_id'],
                'customer_id' => $request->user()?->id,
                'customer_name' => $validated['customer_name'] ?? $request->user()?->name,
                'customer_email' => $validated['customer_email'] ?? $request->user()?->email,
                'customer_phone' => $validated['customer_phone'] ?? null,
                'channel' => $validated['channel'] ?? 'widget',
                'initial_query' => $validated['initial_query'] ?? null,
                'metadata' => $validated['metadata'] ?? null,
                'status' => 'active',
            ]);

            // Send welcome message
            $welcomeMessage = SupportMessage::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'bot',
                'message_text' => $settings->welcome_message ?? SupportBotSettings::defaultWelcomeMessage(),
                'bot_response' => $settings->welcome_message ?? SupportBotSettings::defaultWelcomeMessage(),
                'confidence_score' => 100.0,
            ]);

            $conversation->incrementMessageCount();

            return response()->json([
                'success' => true,
                'data' => [
                    'conversation' => $conversation,
                    'welcome_message' => $welcomeMessage,
                    'bot_available' => true,
                ],
            ], 201);

        } catch (Exception $e) {
            Log::error('Failed to start conversation', [
                'error' => $e->getMessage(),
                'brand_id' => $validated['brand_id'] ?? null,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to start conversation',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Send message and get bot response.
     */
    public function sendMessage(Request $request, string $conversationId): JsonResponse
    {
        $validated = $request->validate([
            'message' => 'required|string|max:2000',
        ]);

        try {
            $conversation = SupportConversation::where('conversation_id', $conversationId)
                ->with('messages')
                ->firstOrFail();

            // Check if conversation is still active
            if (!$conversation->isActive()) {
                return response()->json([
                    'success' => false,
                    'message' => 'This conversation has been closed',
                ], 400);
            }

            $settings = SupportBotSettings::where('brand_id', $conversation->brand_id)->first();

            // Redact PII from customer message
            $piiResult = $this->piiService->redact($validated['message']);

            // Store customer message
            $customerMessage = SupportMessage::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'customer',
                'sender_id' => $request->user()?->id,
                'message_text' => $validated['message'],
                'message_text_redacted' => $piiResult['redacted_text'],
                'detected_pii' => $piiResult['detected_pii'],
            ]);

            $conversation->incrementMessageCount();

            // Check if should escalate based on message count
            if ($settings && $settings->shouldEscalateByMessageCount($conversation->total_messages)) {
                return $this->escalateConversation($conversation, $settings, 'max_messages_reached');
            }

            // Get conversation history for context
            $history = $this->buildConversationHistory($conversation);

            // Generate bot response using RAG
            $ragResponse = $this->ragService->generateAnswer(
                $piiResult['redacted_text'],
                $conversation->brand_id,
                $history
            );

            // Store bot response
            $botMessage = SupportMessage::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'bot',
                'message_text' => $ragResponse['answer'],
                'bot_response' => $ragResponse['answer'],
                'citations' => $ragResponse['citations'],
                'confidence_score' => $ragResponse['confidence_score'],
                'requires_escalation' => $ragResponse['requires_escalation'],
                'metadata' => $ragResponse['metadata'] ?? null,
            ]);

            $conversation->incrementMessageCount();
            $conversation->updateAverageConfidence();

            // Check if should escalate based on confidence
            if ($settings && $ragResponse['requires_escalation'] && 
                $settings->shouldEscalate($ragResponse['confidence_score'])) {
                return $this->escalateConversation($conversation, $settings, 'low_confidence');
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'customer_message' => $customerMessage,
                    'bot_message' => $botMessage,
                    'conversation' => $conversation->fresh(),
                ],
            ]);

        } catch (Exception $e) {
            Log::error('Failed to process message', [
                'error' => $e->getMessage(),
                'conversation_id' => $conversationId,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to process message',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get conversation history.
     */
    public function getConversation(string $conversationId): JsonResponse
    {
        try {
            $conversation = SupportConversation::where('conversation_id', $conversationId)
                ->with(['messages' => function ($query) {
                    $query->orderBy('created_at', 'asc');
                }])
                ->firstOrFail();

            return response()->json([
                'success' => true,
                'data' => $conversation,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Conversation not found',
            ], 404);
        }
    }

    /**
     * Mark message as helpful/not helpful.
     */
    public function provideFeedback(Request $request, int $messageId): JsonResponse
    {
        $validated = $request->validate([
            'helpful' => 'required|boolean',
            'comment' => 'nullable|string|max:500',
        ]);

        try {
            $message = SupportMessage::findOrFail($messageId);

            if ($validated['helpful']) {
                $message->markHelpful($validated['comment'] ?? null);
            } else {
                $message->markNotHelpful($validated['comment'] ?? null);
            }

            return response()->json([
                'success' => true,
                'message' => 'Feedback recorded',
                'data' => $message->fresh(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to record feedback',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Request human agent.
     */
    public function requestHuman(Request $request, string $conversationId): JsonResponse
    {
        $validated = $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        try {
            $conversation = SupportConversation::where('conversation_id', $conversationId)
                ->firstOrFail();

            $settings = SupportBotSettings::where('brand_id', $conversation->brand_id)->first();

            if (!$settings || !$settings->enable_human_handoff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Human handoff is not available',
                ], 400);
            }

            return $this->escalateConversation(
                $conversation,
                $settings,
                $validated['reason'] ?? 'customer_request'
            );

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to request human agent',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Rate conversation.
     */
    public function rateConversation(Request $request, string $conversationId): JsonResponse
    {
        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'feedback' => 'nullable|string|max:1000',
        ]);

        try {
            $conversation = SupportConversation::where('conversation_id', $conversationId)
                ->firstOrFail();

            $conversation->addRating($validated['rating'], $validated['feedback'] ?? null);

            return response()->json([
                'success' => true,
                'message' => 'Rating submitted successfully',
                'data' => $conversation->fresh(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to submit rating',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Close conversation.
     */
    public function closeConversation(string $conversationId): JsonResponse
    {
        try {
            $conversation = SupportConversation::where('conversation_id', $conversationId)
                ->firstOrFail();

            $conversation->close();

            return response()->json([
                'success' => true,
                'message' => 'Conversation closed',
                'data' => $conversation->fresh(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to close conversation',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Build conversation history for context.
     */
    protected function buildConversationHistory(SupportConversation $conversation): array
    {
        $messages = $conversation->messages()
            ->orderBy('created_at', 'asc')
            ->limit(10) // Last 10 messages for context
            ->get();

        $history = [];
        foreach ($messages as $message) {
            if ($message->isFromCustomer()) {
                $history[] = [
                    'role' => 'user',
                    'content' => $message->safe_message_text,
                ];
            } elseif ($message->isFromBot() && $message->bot_response) {
                $history[] = [
                    'role' => 'assistant',
                    'content' => $message->bot_response,
                ];
            }
        }

        return $history;
    }

    /**
     * Escalate conversation to human agent.
     */
    protected function escalateConversation(
        SupportConversation $conversation,
        SupportBotSettings $settings,
        string $reason
    ): JsonResponse {
        $conversation->escalate(null, $reason);

        $escalationMessage = SupportMessage::create([
            'conversation_id' => $conversation->id,
            'sender_type' => 'bot',
            'message_text' => $settings->escalation_message ?? SupportBotSettings::defaultEscalationMessage(),
            'bot_response' => $settings->escalation_message ?? SupportBotSettings::defaultEscalationMessage(),
            'confidence_score' => 0.0,
            'requires_escalation' => true,
            'escalation_reason' => $reason,
        ]);

        $conversation->incrementMessageCount();

        // TODO: Send notification to support team

        return response()->json([
            'success' => true,
            'escalated' => true,
            'data' => [
                'conversation' => $conversation->fresh(),
                'escalation_message' => $escalationMessage,
            ],
        ]);
    }
}