<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\ReputationScore;
use App\Models\ReputationDriver;
use App\Models\ReputationRecommendation;
use App\Models\ReputationAuditLog;
use App\Services\ReputationScoreService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class ReputationController extends Controller
{
    protected ReputationScoreService $reputationService;

    public function __construct(ReputationScoreService $reputationService)
    {
        $this->reputationService = $reputationService;
    }

    /**
     * Display reputation dashboard.
     */
    public function index(Request $request)
    {
        $brand = $request->user()->brand;
        Gate::authorize('view', [ReputationScore::class, $brand]);

        $timeframe = $request->input('timeframe', '30');
        
        // Get latest score
        $latestScore = ReputationScore::latestForBrand($brand->id)
            ->byTimeframe($timeframe)
            ->first();

        // Get historical scores for chart
        $historicalScores = ReputationScore::where('brand_id', $brand->id)
            ->where('status', 'completed')
            ->byTimeframe($timeframe)
            ->orderBy('created_at', 'desc')
            ->limit(30)
            ->get();

        // Get top drivers
        $topPositiveDrivers = $latestScore?->positiveDrivers()->limit(5)->get() ?? collect();
        $topNegativeDrivers = $latestScore?->negativeDrivers()->limit(5)->get() ?? collect();
        $unresolvedIssues = $latestScore?->unresolvedNegativeDrivers()->limit(10)->get() ?? collect();

        // Get recommendations
        $recommendations = $latestScore?->recommendations()
            ->orderByPriority()
            ->get() ?? collect();

        // Get recent audit logs
        $auditLogs = ReputationAuditLog::forBrand($brand->id)
            ->recent(30)
            ->latest()
            ->limit(20)
            ->get();

        return view('brand.reputation.index', compact(
            'brand',
            'latestScore',
            'historicalScores',
            'topPositiveDrivers',
            'topNegativeDrivers',
            'unresolvedIssues',
            'recommendations',
            'auditLogs',
            'timeframe'
        ));
    }

    /**
     * Calculate new reputation score.
     */
    public function calculate(Request $request)
    {
        $brand = $request->user()->brand;
        Gate::authorize('calculate', [ReputationScore::class, $brand]);

        $validated = $request->validate([
            'timeframe' => 'required|in:7,30,90',
            'feed_data' => 'nullable|array',
        ]);

        try {
            $score = $this->reputationService->calculateScore(
                $brand,
                $validated['timeframe'],
                $validated['feed_data'] ?? []
            );

            return redirect()
                ->route('brand.reputation.index', ['timeframe' => $validated['timeframe']])
                ->with('success', 'Reputation score calculated successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to calculate reputation score: ' . $e->getMessage());
        }
    }

    /**
     * Show detailed score breakdown.
     */
    public function show(Request $request, ReputationScore $reputationScore)
    {
        Gate::authorize('view', $reputationScore);

        $reputationScore->load([
            'drivers' => fn($q) => $q->orderByImpact(),
            'recommendations' => fn($q) => $q->orderByPriority(),
            'auditLogs' => fn($q) => $q->latest()->limit(50),
        ]);

        return view('brand.reputation.show', compact('reputationScore'));
    }

    /**
     * Show driver details.
     */
    public function showDriver(Request $request, ReputationDriver $driver)
    {
        Gate::authorize('view', $driver);

        return view('brand.reputation.driver', compact('driver'));
    }

    /**
     * Resolve a negative driver.
     */
    public function resolveDriver(Request $request, ReputationDriver $driver)
    {
        Gate::authorize('resolve', $driver);

        $validated = $request->validate([
            'resolution_note' => 'required|string|max:1000',
        ]);

        $driver->markAsResolved($request->user(), $validated['resolution_note']);

        // Log the resolution
        ReputationAuditLog::create([
            'brand_id' => $driver->brand_id,
            'reputation_score_id' => $driver->reputation_score_id,
            'reputation_driver_id' => $driver->id,
            'user_id' => $request->user()->id,
            'action' => 'driver_resolved',
            'description' => 'Driver marked as resolved',
            'before_state' => ['is_resolved' => false],
            'after_state' => ['is_resolved' => true, 'resolution_note' => $validated['resolution_note']],
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        // Recalculate score
        $this->reputationService->recalculateScore($driver->reputationScore);

        return back()->with('success', 'Driver marked as resolved and score recalculated.');
    }

    /**
     * Update recommendation status.
     */
    public function updateRecommendation(Request $request, ReputationRecommendation $recommendation)
    {
        Gate::authorize('update', $recommendation);

        $validated = $request->validate([
            'status' => 'required|in:pending,in_progress,completed,dismissed',
            'action_taken' => 'nullable|string|max:1000',
            'assigned_to' => 'nullable|exists:users,id',
        ]);

        $beforeState = $recommendation->only(['status', 'action_taken', 'assigned_to']);

        if ($validated['status'] === 'completed') {
            $recommendation->markAsCompleted($validated['action_taken'] ?? null);
        } elseif ($validated['status'] === 'in_progress') {
            $recommendation->markAsInProgress();
        } elseif ($validated['status'] === 'dismissed') {
            $recommendation->dismiss();
        } else {
            $recommendation->update(['status' => $validated['status']]);
        }

        if (isset($validated['assigned_to'])) {
            $recommendation->update(['assigned_to' => $validated['assigned_to']]);
        }

        // Log the update
        ReputationAuditLog::create([
            'brand_id' => $recommendation->brand_id,
            'reputation_score_id' => $recommendation->reputation_score_id,
            'user_id' => $request->user()->id,
            'action' => 'settings_updated',
            'description' => 'Recommendation status updated',
            'before_state' => $beforeState,
            'after_state' => $recommendation->only(['status', 'action_taken', 'assigned_to']),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return back()->with('success', 'Recommendation updated successfully.');
    }

    /**
     * Get score history data for charts.
     */
    public function history(Request $request)
    {
        $brand = $request->user()->brand;
        Gate::authorize('view', [ReputationScore::class, $brand]);

        $timeframe = $request->input('timeframe', '30');
        $limit = $request->input('limit', 30);

        $scores = ReputationScore::where('brand_id', $brand->id)
            ->where('status', 'completed')
            ->byTimeframe($timeframe)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get()
            ->reverse()
            ->values();

        return response()->json([
            'labels' => $scores->pluck('created_at')->map(fn($date) => $date->format('M d')),
            'overall' => $scores->pluck('score'),
            'sentiment' => $scores->pluck('sentiment_score'),
            'responsiveness' => $scores->pluck('responsiveness_score'),
            'trust' => $scores->pluck('trust_score'),
            'compliance' => $scores->pluck('compliance_score'),
            'growth' => $scores->pluck('growth_score'),
        ]);
    }

    /**
     * Get drivers by category.
     */
    public function driversByCategory(Request $request, ReputationScore $reputationScore)
    {
        Gate::authorize('view', $reputationScore);

        $category = $request->input('category');
        $type = $request->input('type');

        $query = $reputationScore->drivers();

        if ($category) {
            $query->byCategory($category);
        }

        if ($type) {
            $query->where('type', $type);
        }

        $drivers = $query->orderByImpact()->paginate(20);

        return response()->json($drivers);
    }

    /**
     * Export reputation report.
     */
    public function export(Request $request, ReputationScore $reputationScore)
    {
        Gate::authorize('view', $reputationScore);

        $format = $request->input('format', 'pdf');

        // TODO: Implement PDF/Excel export
        // This would generate a comprehensive report with all score details

        return back()->with('info', 'Export functionality coming soon.');
    }

    /**
     * Get audit log.
     */
    public function auditLog(Request $request)
    {
        $brand = $request->user()->brand;
        Gate::authorize('view', [ReputationScore::class, $brand]);

        $logs = ReputationAuditLog::forBrand($brand->id)
            ->with(['user', 'reputationScore', 'reputationDriver'])
            ->latest()
            ->paginate(50);

        return view('brand.reputation.audit-log', compact('logs'));
    }

    /**
     * Get alerts.
     */
    public function alerts(Request $request)
    {
        $brand = $request->user()->brand;
        Gate::authorize('view', [ReputationScore::class, $brand]);

        $alerts = ReputationScore::where('brand_id', $brand->id)
            ->withAlerts()
            ->latest()
            ->paginate(20);

        return view('brand.reputation.alerts', compact('alerts'));
    }

    /**
     * Dismiss alert.
     */
    public function dismissAlert(Request $request, ReputationScore $reputationScore)
    {
        Gate::authorize('update', $reputationScore);

        $reputationScore->update(['alert_triggered' => false]);

        ReputationAuditLog::create([
            'brand_id' => $reputationScore->brand_id,
            'reputation_score_id' => $reputationScore->id,
            'user_id' => $request->user()->id,
            'action' => 'settings_updated',
            'description' => 'Alert dismissed',
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return back()->with('success', 'Alert dismissed.');
    }
}