<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Jobs\ProcessPerformanceForecast;
use App\Models\Campaign;
use App\Models\PerformanceForecast;
use App\Models\SocialAccount;
use App\Services\PerformanceForecastService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class PerformanceForecastController extends Controller
{
    protected $forecastService;

    public function __construct(PerformanceForecastService $forecastService)
    {
        $this->forecastService = $forecastService;
    }

    /**
     * Display the performance forecasting dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $brand = auth()->user()->brand;
        
        if (!$brand) {
            return redirect()->route('dashboard')->with('error', 'Brand not found');
        }

        // Get recent forecasts
        $forecasts = PerformanceForecast::forBrand($brand->id)
            ->with('creator')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Get available campaigns and social accounts for filtering
        $campaigns = Campaign::where('brand_id', $brand->id)
            ->where('status', '!=', 'cancelled')
            ->get();
            
        $socialAccounts = SocialAccount::where('brand_id', $brand->id)->get();

        return view('brand.forecasts.index', compact('forecasts', 'campaigns', 'socialAccounts'));
    }

    /**
     * Show the form for creating a new forecast.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $brand = auth()->user()->brand;
        
        if (!$brand) {
            return redirect()->route('dashboard')->with('error', 'Brand not found');
        }

        $campaigns = Campaign::where('brand_id', $brand->id)
            ->where('status', '!=', 'cancelled')
            ->get();
            
        $socialAccounts = SocialAccount::where('brand_id', $brand->id)->get();

        return view('brand.forecasts.create', compact('campaigns', 'socialAccounts'));
    }

    /**
     * Store a newly created forecast.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $brand = auth()->user()->brand;
        
        if (!$brand) {
            return response()->json(['error' => 'Brand not found'], 404);
        }

        $validated = $request->validate([
            'campaign_ids' => 'nullable|array',
            'campaign_ids.*' => 'exists:campaigns,id',
            'social_account_ids' => 'nullable|array',
            'social_account_ids.*' => 'exists:social_accounts,id',
            'start_date' => 'required|date|before:end_date',
            'end_date' => 'required|date|before_or_equal:today',
            'forecast_days' => 'required|integer|min:7|max:90',
            'scenario' => 'required|in:conservative,balanced,aggressive',
        ]);

        // Verify campaigns belong to brand
        if (!empty($validated['campaign_ids'])) {
            $campaignCount = Campaign::where('brand_id', $brand->id)
                ->whereIn('id', $validated['campaign_ids'])
                ->count();
                
            if ($campaignCount !== count($validated['campaign_ids'])) {
                return response()->json(['error' => 'Invalid campaign selection'], 403);
            }
        }

        // Verify social accounts belong to brand
        if (!empty($validated['social_account_ids'])) {
            $accountCount = SocialAccount::where('brand_id', $brand->id)
                ->whereIn('id', $validated['social_account_ids'])
                ->count();
                
            if ($accountCount !== count($validated['social_account_ids'])) {
                return response()->json(['error' => 'Invalid social account selection'], 403);
            }
        }

        // Create forecast
        $forecast = $this->forecastService->generateForecast($brand, $validated);

        // Dispatch job for batch processing
        ProcessPerformanceForecast::dispatch($forecast);

        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'Forecast created successfully',
                'forecast' => $forecast,
            ], 201);
        }

        return redirect()->route('brand.forecasts.show', $forecast)
            ->with('success', 'Forecast is being generated. This may take a few moments.');
    }

    /**
     * Display the specified forecast.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\Response
     */
    public function show(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            abort(403, 'Unauthorized access to forecast');
        }

        $forecast->load('creator');

        // Get associated campaigns and social accounts
        $campaigns = $forecast->campaigns();
        $socialAccounts = $forecast->socialAccounts();

        return view('brand.forecasts.show', compact('forecast', 'campaigns', 'socialAccounts'));
    }

    /**
     * Get forecast data as JSON for charts.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\JsonResponse
     */
    public function data(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        if (!$forecast->isCompleted()) {
            return response()->json([
                'status' => $forecast->status,
                'message' => 'Forecast is still processing',
            ], 202);
        }

        return response()->json([
            'status' => 'completed',
            'predicted_timeseries' => $forecast->predicted_timeseries,
            'error_bands' => $forecast->error_bands,
            'roi_estimates' => $forecast->roi_estimates,
            'insights' => [
                'text' => $forecast->insights_text,
                'actionable' => $forecast->getActionableInsights(),
            ],
            'kpis' => $forecast->getPredictedKPIs(),
        ]);
    }

    /**
     * Compare multiple forecast scenarios.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function compareScenarios(Request $request)
    {
        $brand = auth()->user()->brand;
        
        if (!$brand) {
            return response()->json(['error' => 'Brand not found'], 404);
        }

        $validated = $request->validate([
            'campaign_ids' => 'nullable|array',
            'campaign_ids.*' => 'exists:campaigns,id',
            'social_account_ids' => 'nullable|array',
            'social_account_ids.*' => 'exists:social_accounts,id',
            'start_date' => 'required|date|before:end_date',
            'end_date' => 'required|date|before_or_equal:today',
            'forecast_days' => 'required|integer|min:7|max:90',
        ]);

        // Generate forecasts for all scenarios
        $comparisons = $this->forecastService->compareScenarios($brand, $validated);

        return response()->json([
            'message' => 'Scenario comparison completed',
            'scenarios' => $comparisons,
        ]);
    }

    /**
     * Get forecast status.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\JsonResponse
     */
    public function status(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        return response()->json([
            'id' => $forecast->id,
            'status' => $forecast->status,
            'completed_at' => $forecast->completed_at,
            'error_message' => $forecast->error_message,
        ]);
    }

    /**
     * Delete a forecast.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\Response
     */
    public function destroy(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            abort(403, 'Unauthorized access to forecast');
        }

        $forecast->delete();

        if (request()->expectsJson()) {
            return response()->json(['message' => 'Forecast deleted successfully']);
        }

        return redirect()->route('brand.forecasts.index')
            ->with('success', 'Forecast deleted successfully');
    }

    /**
     * Export forecast data.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\Response
     */
    public function export(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            abort(403, 'Unauthorized access to forecast');
        }

        if (!$forecast->isCompleted()) {
            return redirect()->back()->with('error', 'Forecast is not yet completed');
        }

        $data = [
            'forecast_id' => $forecast->id,
            'created_at' => $forecast->created_at->toDateTimeString(),
            'scenario' => $forecast->scenario,
            'forecast_period' => [
                'start_date' => $forecast->start_date->toDateString(),
                'end_date' => $forecast->end_date->toDateString(),
                'forecast_days' => $forecast->forecast_days,
            ],
            'predicted_timeseries' => $forecast->predicted_timeseries,
            'error_bands' => $forecast->error_bands,
            'roi_estimates' => $forecast->roi_estimates,
            'insights' => [
                'text' => $forecast->insights_text,
                'actionable' => $forecast->getActionableInsights(),
            ],
            'kpis' => $forecast->getPredictedKPIs(),
        ];

        $filename = "forecast_{$forecast->id}_" . now()->format('Y-m-d') . ".json";

        return response()->json($data)
            ->header('Content-Type', 'application/json')
            ->header('Content-Disposition', "attachment; filename=\"{$filename}\"");
    }

    /**
     * Get historical vs predicted comparison.
     *
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return \Illuminate\Http\JsonResponse
     */
    public function comparison(PerformanceForecast $forecast)
    {
        // Authorize access
        if ($forecast->brand_id !== auth()->user()->brand_id) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        if (!$forecast->isCompleted()) {
            return response()->json([
                'error' => 'Forecast is not yet completed',
            ], 400);
        }

        return response()->json([
            'historical' => $forecast->input_metrics,
            'predicted' => $forecast->predicted_timeseries,
            'error_bands' => $forecast->error_bands,
        ]);
    }
}