<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\CompetitorBrand;
use App\Models\CompetitorInsight;
use App\Models\CompetitorAlert;
use App\Services\CompetitorDataFetchService;
use App\Services\CompetitorAnalysisService;
use App\Services\CompetitorAIInsightsService;
use App\Services\CompetitorAlertService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class CompetitorInsightsController extends Controller
{
    protected CompetitorDataFetchService $dataFetchService;
    protected CompetitorAnalysisService $analysisService;
    protected CompetitorAIInsightsService $aiService;
    protected CompetitorAlertService $alertService;

    public function __construct(
        CompetitorDataFetchService $dataFetchService,
        CompetitorAnalysisService $analysisService,
        CompetitorAIInsightsService $aiService,
        CompetitorAlertService $alertService
    ) {
        $this->middleware(['auth', 'role:brand_admin']);
        $this->dataFetchService = $dataFetchService;
        $this->analysisService = $analysisService;
        $this->aiService = $aiService;
        $this->alertService = $alertService;
    }

    /**
     * Display competitor insights dashboard.
     */
    public function index(Request $request)
    {
        $brand = Auth::user()->brand;
        
        $competitors = CompetitorBrand::where('brand_id', $brand->id)
            ->active()
            ->with(['insights' => function($query) {
                $query->completed()->latest()->limit(1);
            }])
            ->get();

        $recentAlerts = $this->alertService->getUnreadAlerts($brand->id, 5);

        return view('brand.competitor-insights.index', compact('competitors', 'recentAlerts'));
    }

    /**
     * Show competitor list management.
     */
    public function competitors(Request $request)
    {
        $brand = Auth::user()->brand;
        
        $competitors = CompetitorBrand::where('brand_id', $brand->id)
            ->withCount('posts')
            ->paginate(15);

        return view('brand.competitor-insights.competitors', compact('competitors'));
    }

    /**
     * Store a new competitor.
     */
    public function storeCompetitor(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'industry' => 'nullable|string|max:100',
            'social_handles' => 'required|array',
            'social_handles.*' => 'required|string',
        ]);

        $brand = Auth::user()->brand;

        $competitor = CompetitorBrand::create([
            'brand_id' => $brand->id,
            'tenant_id' => $brand->tenant_id,
            'name' => $validated['name'],
            'slug' => \Str::slug($validated['name']),
            'description' => $validated['description'] ?? null,
            'industry' => $validated['industry'] ?? null,
            'social_handles' => $validated['social_handles'],
            'is_active' => true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Competitor added successfully',
            'competitor' => $competitor,
        ]);
    }

    /**
     * Update a competitor.
     */
    public function updateCompetitor(Request $request, CompetitorBrand $competitor)
    {
        $this->authorize('update', $competitor);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'industry' => 'nullable|string|max:100',
            'social_handles' => 'sometimes|array',
            'is_active' => 'sometimes|boolean',
        ]);

        $competitor->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Competitor updated successfully',
            'competitor' => $competitor,
        ]);
    }

    /**
     * Delete a competitor.
     */
    public function destroyCompetitor(CompetitorBrand $competitor)
    {
        $this->authorize('delete', $competitor);

        $competitor->delete();

        return response()->json([
            'success' => true,
            'message' => 'Competitor deleted successfully',
        ]);
    }

    /**
     * Show detailed insights for a specific competitor.
     */
    public function show(CompetitorBrand $competitor, Request $request)
    {
        $this->authorize('view', $competitor);

        $dateRange = $request->input('date_range', '30');
        $endDate = Carbon::now();
        $startDate = Carbon::now()->subDays((int)$dateRange);

        // Get or create insight
        $insight = CompetitorInsight::where('competitor_brand_id', $competitor->id)
            ->where('brand_id', $competitor->brand_id)
            ->where('start_date', $startDate->toDateString())
            ->where('end_date', $endDate->toDateString())
            ->first();

        if (!$insight || $insight->status === 'failed') {
            // Generate new insight
            $insight = $this->analysisService->generateInsights(
                $competitor->brand,
                $competitor,
                $startDate,
                $endDate
            );

            // Generate AI recommendations and summary
            $recommendations = $this->aiService->generateRecommendations($insight);
            $summary = $this->aiService->generateSummary($insight);

            $insight->update([
                'recommendations' => $recommendations,
                'summary' => $summary,
            ]);
        }

        $alerts = CompetitorAlert::where('competitor_brand_id', $competitor->id)
            ->where('brand_id', $competitor->brand_id)
            ->active()
            ->recent(30)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('brand.competitor-insights.show', compact('competitor', 'insight', 'alerts'));
    }

    /**
     * Get insights data as JSON.
     */
    public function insightsData(CompetitorBrand $competitor, Request $request)
    {
        $this->authorize('view', $competitor);

        $dateRange = $request->input('date_range', '30');
        $endDate = Carbon::now();
        $startDate = Carbon::now()->subDays((int)$dateRange);

        $insight = CompetitorInsight::where('competitor_brand_id', $competitor->id)
            ->where('brand_id', $competitor->brand_id)
            ->where('start_date', $startDate->toDateString())
            ->where('end_date', $endDate->toDateString())
            ->completed()
            ->first();

        if (!$insight) {
            return response()->json([
                'success' => false,
                'message' => 'No insights available for this period',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'insight' => $insight,
        ]);
    }

    /**
     * Sync competitor data manually.
     */
    public function syncCompetitor(CompetitorBrand $competitor)
    {
        $this->authorize('update', $competitor);

        try {
            $results = $this->dataFetchService->fetchCompetitorPosts($competitor);
            
            // Check for alerts after syncing
            $this->alertService->checkForAlerts($competitor);

            return response()->json([
                'success' => true,
                'message' => 'Competitor data synced successfully',
                'results' => $results,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to sync competitor data: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get alerts for the brand.
     */
    public function alerts(Request $request)
    {
        $brand = Auth::user()->brand;
        
        $query = CompetitorAlert::where('brand_id', $brand->id)
            ->with(['competitorBrand', 'competitorPost']);

        // Filter by status
        if ($request->has('status')) {
            if ($request->status === 'unread') {
                $query->unread();
            } elseif ($request->status === 'read') {
                $query->read();
            }
        }

        // Filter by severity
        if ($request->has('severity')) {
            $query->severity($request->severity);
        }

        // Filter by competitor
        if ($request->has('competitor_id')) {
            $query->forCompetitor($request->competitor_id);
        }

        $alerts = $query->active()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('brand.competitor-insights.alerts', compact('alerts'));
    }

    /**
     * Mark alert as read.
     */
    public function markAlertRead(CompetitorAlert $alert)
    {
        $this->authorize('update', $alert);

        $alert->markAsRead();

        return response()->json([
            'success' => true,
            'message' => 'Alert marked as read',
        ]);
    }

    /**
     * Dismiss alert.
     */
    public function dismissAlert(CompetitorAlert $alert)
    {
        $this->authorize('update', $alert);

        $alert->dismiss();

        return response()->json([
            'success' => true,
            'message' => 'Alert dismissed',
        ]);
    }

    /**
     * Bulk mark alerts as read.
     */
    public function bulkMarkRead(Request $request)
    {
        $validated = $request->validate([
            'alert_ids' => 'required|array',
            'alert_ids.*' => 'exists:competitor_alerts,id',
        ]);

        $count = $this->alertService->markAlertsAsRead($validated['alert_ids']);

        return response()->json([
            'success' => true,
            'message' => "{$count} alerts marked as read",
        ]);
    }

    /**
     * Export insights report.
     */
    public function exportInsights(CompetitorBrand $competitor, Request $request)
    {
        $this->authorize('view', $competitor);

        $dateRange = $request->input('date_range', '30');
        $endDate = Carbon::now();
        $startDate = Carbon::now()->subDays((int)$dateRange);

        $insight = CompetitorInsight::where('competitor_brand_id', $competitor->id)
            ->where('brand_id', $competitor->brand_id)
            ->where('start_date', $startDate->toDateString())
            ->where('end_date', $endDate->toDateString())
            ->completed()
            ->first();

        if (!$insight) {
            return response()->json([
                'success' => false,
                'message' => 'No insights available to export',
            ], 404);
        }

        // Generate export data
        $exportData = [
            'competitor' => $competitor->name,
            'period' => "{$startDate->format('M d, Y')} - {$endDate->format('M d, Y')}",
            'summary' => $insight->summary,
            'metrics' => $insight->competitor_metrics,
            'comparison' => $insight->comparison_data,
            'recommendations' => $insight->recommendations,
            'top_posts' => $insight->top_posts,
        ];

        return response()->json([
            'success' => true,
            'data' => $exportData,
        ]);
    }

    /**
     * Compare multiple competitors.
     */
    public function compare(Request $request)
    {
        $validated = $request->validate([
            'competitor_ids' => 'required|array|min:2|max:5',
            'competitor_ids.*' => 'exists:competitor_brands,id',
            'date_range' => 'nullable|integer|min:7|max:90',
        ]);

        $brand = Auth::user()->brand;
        $dateRange = $validated['date_range'] ?? 30;
        $endDate = Carbon::now();
        $startDate = Carbon::now()->subDays($dateRange);

        $competitors = CompetitorBrand::whereIn('id', $validated['competitor_ids'])
            ->where('brand_id', $brand->id)
            ->with(['insights' => function($query) use ($startDate, $endDate) {
                $query->where('start_date', $startDate->toDateString())
                      ->where('end_date', $endDate->toDateString())
                      ->completed();
            }])
            ->get();

        return view('brand.competitor-insights.compare', compact('competitors', 'startDate', 'endDate'));
    }
}